<?php
/*
	+-----------------------------------------------------------------------------+
	| ILIAS open source                                                           |
	+-----------------------------------------------------------------------------+
	| Copyright (c) 1998-2001 ILIAS open source, University of Cologne            |
	|                                                                             |
	| This program is free software; you can redistribute it and/or               |
	| modify it under the terms of the GNU General Public License                 |
	| as published by the Free Software Foundation; either version 2              |
	| of the License, or (at your option) any later version.                      |
	|                                                                             |
	| This program is distributed in the hope that it will be useful,             |
	| but WITHOUT ANY WARRANTY; without even the implied warranty of              |
	| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               |
	| GNU General Public License for more details.                                |
	|                                                                             |
	| You should have received a copy of the GNU General Public License           |
	| along with this program; if not, write to the Free Software                 |
	| Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. |
	+-----------------------------------------------------------------------------+
*/

/**
* Class ilObjFileAccessSettingsGUI
*
* @author Werner Randelshofer, Hochschule Luzern, werner.randelshofer@hslu.ch
*
* @version $Id$
*
* @ilCtrl_Calls ilObjFileAccessSettingsGUI: ilPermissionGUI, ilFMSettingsGUI
*
* @extends ilObjectGUI
* @package webdav
*/

include_once "./Services/Object/classes/class.ilObjectGUI.php";

class ilObjFileAccessSettingsGUI extends ilObjectGUI
{
	private $disk_quota_obj;

	/**
	* Constructor
	* @access public
	*/
	function ilObjFileAccessSettingsGUI($a_data,$a_id,$a_call_by_reference)
	{
		global $tree;

		$this->type = "facs";
		$this->ilObjectGUI($a_data,$a_id,$a_call_by_reference, false);
		$this->folderSettings = new ilSetting('fold');

		// Load the disk quota settings object
		require_once 'Services/WebDAV/classes/class.ilObjDiskQuotaSettings.php';
		$this->disk_quota_obj = new ilObjDiskQuotaSettings($a_id, $a_call_by_reference);
		$this->disk_quota_obj->read();
	}
	/**
	 * Execute command
	 *
	 * @access public
	 *
	 */
	public function executeCommand()
	{
		global $rbacsystem,$ilErr,$ilAccess, $ilias, $lng;

		$next_class = $this->ctrl->getNextClass($this);
		$cmd = $this->ctrl->getCmd();

		$this->prepareOutput();

		if(!$ilAccess->checkAccess('read','',$this->object->getRefId()))
		{
			$ilias->raiseError($lng->txt('no_permission'),$ilias->error_obj->MESSAGE);
		}

		switch($next_class)
		{
			case 'ilpermissiongui':
				$this->tabs_gui->setTabActive('perm_settings');
				include_once("Services/AccessControl/classes/class.ilPermissionGUI.php");
				$perm_gui =& new ilPermissionGUI($this);
				$ret =& $this->ctrl->forwardCommand($perm_gui);
				break;

			case 'ilfmsettingsgui':
				$this->tabs_gui->setTabActive('fm_settings_tab');
				include_once './Services/WebServices/FileManager/classes/class.ilFMSettingsGUI.php';
				$fmg = new ilFMSettingsGUI($this);
				$this->ctrl->forwardCommand($fmg);
				break;

			default:
				if(!$cmd || $cmd == 'view')
				{
					$cmd = "editDownloadingSettings";
				}

				$this->$cmd();
				break;
		}
		return true;
	}

	/**
	 * Get tabs
	 *
	 * @access public
	 *
	 */
	public function getAdminTabs()
	{
		global $rbacsystem, $ilAccess;

		$GLOBALS['lng']->loadLanguageModule('fm');

		if ($rbacsystem->checkAccess("visible,read",$this->object->getRefId()))
		{
			$this->tabs_gui->addTarget('downloading_settings',
				$this->ctrl->getLinkTarget($this, "editDownloadingSettings"),
				array("editDownloadingSettings", "view"));

			$this->tabs_gui->addTarget('upload_settings',
				$this->ctrl->getLinkTarget($this, "editUploadSettings"),
				array("editUploadSettings", "view"));

			$this->tabs_gui->addTarget('preview_settings',
				$this->ctrl->getLinkTarget($this, "editPreviewSettings"),
				array("editPreviewSettings", "view"));

			$this->tabs_gui->addTarget('webdav',
				$this->ctrl->getLinkTarget($this, "editWebDAVSettings"),
				array("editWebDAVSettings", "view"));

			$this->tabs_gui->addTarget(
				'fm_settings_tab',
				$this->ctrl->getLinkTargetByClass('ilFMSettingsGUI','settings'),
				array(),
				'ilfmsettingsgui'
			);

			$this->tabs_gui->addTarget("disk_quota",
				$this->ctrl->getLinkTarget($this, "editDiskQuotaSettings"),
				array("editDiskQuota", "view"));
		}
		if ($rbacsystem->checkAccess('edit_permission',$this->object->getRefId()))
		{
			$this->tabs_gui->addTarget("perm_settings",
				$this->ctrl->getLinkTargetByClass('ilpermissiongui',"perm"),
				array(),'ilpermissiongui');
		}
	}

	/**
	* Edit settings.
	*/
	public function editDownloadingSettings()
	{
		global $rbacsystem, $ilErr, $ilTabs;

		$this->tabs_gui->setTabActive('downloading_settings');

		if (! $rbacsystem->checkAccess("visible,read",$this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}

		global $tpl, $ilCtrl, $lng, $tree, $settings;

		require_once("./Services/Form/classes/class.ilPropertyFormGUI.php");
		require_once("./Services/Form/classes/class.ilCheckboxInputGUI.php");
		require_once("./Services/Form/classes/class.ilRadioGroupInputGUI.php");
		require_once("./Services/Form/classes/class.ilRadioOption.php");
		require_once("./Services/Form/classes/class.ilTextAreaInputGUI.php");

		$form = new ilPropertyFormGUI();
		$form->setFormAction($ilCtrl->getFormAction($this));
		$form->setTitle($lng->txt("settings"));
		
		// Backwards compatibility with ILIAS 3.9: Use the name of the
		// uploaded file as the filename for the downloaded file instead
		// of the title of the file object.
		$dl_prop = new ilCheckboxInputGUI($lng->txt("download_with_uploaded_filename"), "download_with_uploaded_filename");
		$dl_prop->setValue('1');
		// default value should reflect previous behaviour (-> 0)
		$dl_prop->setChecked($this->object->isDownloadWithUploadedFilename() == 1);
		$dl_prop->setInfo($lng->txt('download_with_uploaded_filename_info'));
		$form->addItem($dl_prop);

		// Show download action for folder
		$dl_prop = new ilCheckboxInputGUI($lng->txt("enable_download_folder"), "enable_download_folder");
		$dl_prop->setValue('1');
		// default value should reflect previous behaviour (-> 0)
		$dl_prop->setChecked($this->folderSettings->get("enable_download_folder", 0) == 1);
		$dl_prop->setInfo($lng->txt('enable_download_folder_info'));
		$form->addItem($dl_prop);

		// multi download
		$dl_prop = new ilCheckboxInputGUI($lng->txt("enable_multi_download"), "enable_multi_download");
		$dl_prop->setValue('1');
		// default value should reflect previous behaviour (-> 0)
		$dl_prop->setChecked($this->folderSettings->get("enable_multi_download", 0) == 1);
		$dl_prop->setInfo($lng->txt('enable_multi_download_info'));
		$form->addItem($dl_prop);
		
		// Inline file extensions
		$tai_prop = new ilTextAreaInputGUI($lng->txt('inline_file_extensions'), 'inline_file_extensions');
		$tai_prop->setValue($this->object->getInlineFileExtensions());
		$tai_prop->setInfo($lng->txt('inline_file_extensions_info'));
		$tai_prop->setCols(80);
		$tai_prop->setRows(5);
		$form->addItem($tai_prop);


		// command buttons
		$form->addCommandButton('saveDownloadingSettings', $lng->txt('save'));
		$form->addCommandButton('view', $lng->txt('cancel'));

		$tpl->setContent($form->getHTML());
	}

	/**
	* Save settings
	*/
	public function saveDownloadingSettings()
	{
		global $rbacsystem, $ilErr, $ilCtrl, $lng;

		if (! $rbacsystem->checkAccess("write",$this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}

		$this->object->setDownloadWithUploadedFilename(ilUtil::stripSlashes($_POST['download_with_uploaded_filename']));
		$this->object->setInlineFileExtensions(ilUtil::stripSlashes($_POST['inline_file_extensions']));
		$this->object->update();

		$this->folderSettings->set("enable_download_folder", $_POST["enable_download_folder"] == 1);
		$this->folderSettings->set("enable_multi_download", $_POST["enable_multi_download"] == 1);

		ilUtil::sendInfo($lng->txt('settings_saved'),true);
		$ilCtrl->redirect($this, "editDownloadingSettings");
	}

	/**
	* Edit settings.
	*/
	public function editWebDAVSettings()
	{
		global $rbacsystem, $ilErr, $ilTabs;
		global $tpl, $ilCtrl, $lng, $tree, $settings;


		$this->tabs_gui->setTabActive('webdav');

		if (! $rbacsystem->checkAccess("visible,read",$this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}

		require_once("./Services/Form/classes/class.ilPropertyFormGUI.php");
		require_once("./Services/Form/classes/class.ilCheckboxInputGUI.php");
		require_once("./Services/Form/classes/class.ilRadioGroupInputGUI.php");
		require_once("./Services/Form/classes/class.ilRadioOption.php");
		require_once("./Services/Form/classes/class.ilTextAreaInputGUI.php");
		require_once("./Services/WebDAV/classes/class.ilDAVServer.php");

		$form = new ilPropertyFormGUI();
		$form->setFormAction($ilCtrl->getFormAction($this));
		$form->setTitle($lng->txt("settings"));

		// Enable webdav
		$ilDAVServer = ilDAVServer::getInstance();
		$isPearAuthHTTPInstalled = @include_once("Auth/HTTP.php");
		$cb_prop = new ilCheckboxInputGUI($lng->txt("enable_webdav"), "enable_webdav");
		$cb_prop->setValue('1');
		$cb_prop->setChecked($this->object->isWebdavEnabled() && $isPearAuthHTTPInstalled);
		$cb_prop->setDisabled(! $isPearAuthHTTPInstalled);
		$cb_prop->setInfo($isPearAuthHTTPInstalled ?
			sprintf($lng->txt('enable_webdav_info'),$ilDAVServer->getMountURI($tree->getRootId(),0,null,null,true)) :
			$lng->txt('webdav_pear_auth_http_needed')
		);
		$form->addItem($cb_prop);

		// Webdav help text
		if ($isPearAuthHTTPInstalled)
		{
			$rgi_prop = new ilRadioGroupInputGUI($lng->txt('webfolder_instructions'), 'custom_webfolder_instructions_choice');
			$rgi_prop->addOption(new ilRadioOption($lng->txt('use_default_instructions'), 'default'));
			$rgi_prop->addOption(new ilRadioOption($lng->txt('use_customized_instructions'), 'custom'));
			$rgi_prop->setValue($this->object->isCustomWebfolderInstructionsEnabled() ? 'custom':'default');
			$rgi_prop->setDisabled(! $isPearAuthHTTPInstalled);
			$form->addItem($rgi_prop);
			$tai_prop = new ilTextAreaInputGUI('', 'custom_webfolder_instructions');
			$tai_prop->setValue($this->object->getCustomWebfolderInstructions());
			$tai_prop->setInfo($lng->txt("webfolder_instructions_info"));
			$tai_prop->setCols(80);
			$tai_prop->setRows(20);
			$tai_prop->setDisabled(! $isPearAuthHTTPInstalled);
			$form->addItem($tai_prop);
		}

		// command buttons
		$form->addCommandButton('saveWebDAVSettings', $lng->txt('save'));
		$form->addCommandButton('view', $lng->txt('cancel'));

		$tpl->setContent($form->getHTML());
	}

	/**
	* Save settings
	*/
	public function saveWebDAVSettings()
	{
		global $rbacsystem, $ilErr, $ilCtrl, $lng;

		if (! $rbacsystem->checkAccess("write",$this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}

		$this->object->setWebdavEnabled($_POST['enable_webdav'] == '1');
//		$this->object->setWebdavActionsVisible($_POST['webdav_actions_visible'] == '1');
		$this->object->setCustomWebfolderInstructionsEnabled($_POST['custom_webfolder_instructions_choice']=='custom');
		$this->object->setCustomWebfolderInstructions(ilUtil::stripSlashes($_POST['custom_webfolder_instructions'], false));
		$this->object->update();
		
		ilUtil::sendInfo($lng->txt('settings_saved'),true);
		$ilCtrl->redirect($this, "editWebDAVSettings");
	}
	
	/**
	* called by prepare output
	*/
	function setTitleAndDescription()
	{
		parent::setTitleAndDescription();
		$this->tpl->setDescription($this->object->getDescription());
	}


	// DISK QUOTA --------------------------------------------------------------
	/**
	* Add disk quota subtabs
	*/
	function addDiskQuotaSubtabs($a_active_subtab)
	{
		global $ilCtrl, $ilTabs;

		include_once("./Services/COPage/classes/class.ilPageEditorSettings.php");

		$ilTabs->addSubTabTarget("settings",
			 $ilCtrl->getLinkTarget($this, "editDiskQuotaSettings"),
			 array("editDiskQuotaSettings"));

		require_once 'Services/WebDAV/classes/class.ilDiskQuotaActivationChecker.php';
		if (ilDiskQuotaActivationChecker::_isActive())
		{
			$ilTabs->addSubTabTarget("disk_quota_report",
				 $ilCtrl->getLinkTarget($this, "viewDiskQuotaReport"),
				 array("viewDiskQuotaReport"));
		}

		$ilTabs->addSubTabTarget("disk_quota_reminder_mail",
			 $ilCtrl->getLinkTarget($this, "editDiskQuotaMailTemplate"),
			 array("editDiskQuotaMailTemplate"));

		$ilTabs->setSubTabActive($a_active_subtab);
	}


	/**
	* Edit disk quota settings.
	*/
	public function editDiskQuotaSettings()
	{
		global $rbacsystem, $ilErr, $ilSetting, $tpl, $lng, $ilCtrl;


		if (! $rbacsystem->checkAccess("visible,read",$this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}

		$this->tabs_gui->setTabActive('disk_quota');
		$this->addDiskQuotaSubtabs('settings');

		require_once("./Services/Form/classes/class.ilPropertyFormGUI.php");
		require_once("./Services/Form/classes/class.ilCheckboxInputGUI.php");
		require_once("./Services/Form/classes/class.ilRadioGroupInputGUI.php");
		require_once("./Services/Form/classes/class.ilRadioOption.php");
		require_once("./Services/Form/classes/class.ilTextAreaInputGUI.php");
		require_once("./Services/WebDAV/classes/class.ilDAVServer.php");
		
		$lng->loadLanguageModule("file");

		$form = new ilPropertyFormGUI();
		$form->setFormAction($ilCtrl->getFormAction($this));
		$form->setTitle($lng->txt("settings"));
		
		include_once "Services/Administration/classes/class.ilAdministrationSettingsFormHandler.php";
		ilAdministrationSettingsFormHandler::addFieldsToForm(
			ilAdministrationSettingsFormHandler::FORM_FILES_QUOTA, 
			$form,
			$this
		);

		/*
		// command buttons
		$form->addCommandButton('saveDiskQuotaSettings', $lng->txt('save'));
		$form->addCommandButton('editDiskQuotaSettings', $lng->txt('cancel'));
		*/
		
		$tpl->setContent($form->getHTML());
	}

	/**
	* Save disk quota settings.
	*/
	public function saveDiskQuotaSettings()
	{
		global $rbacsystem, $ilErr, $ilCtrl, $lng;

		if (! $rbacsystem->checkAccess("write",$this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}

		// ilUtil::sendInfo($lng->txt('settings_saved'),true);
		$ilCtrl->redirect($this, "editDiskQuotaSettings");
	}

	/**
	* The disk quota report list shows user accounts, their disk quota and their
    * disk usage, as well as the last time a reminder was sent.
	*/
	public function viewDiskQuotaReport()
	{
		global $rbacsystem, $ilErr, $ilSetting, $lng;

		if (! $rbacsystem->checkAccess("visible,read",$this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}

		$this->tabs_gui->setTabActive('disk_quota');
		$this->addDiskQuotaSubtabs('disk_quota_report');

		// nothing to do if disk quota is not active
		require_once 'Services/WebDAV/classes/class.ilDiskQuotaActivationChecker.php';
		if (! ilDiskQuotaActivationChecker::_isActive())
		{
			return;
		}

		// get the form
		$this->tpl->addBlockfile('ADM_CONTENT','adm_content','tpl.disk_quota_report.html',
			"Services/WebDAV");

		// get the date of the last update
		require_once("./Services/WebDAV/classes/class.ilDiskQuotaChecker.php");
		$last_update = ilDiskQuotaChecker::_lookupDiskUsageReportLastUpdate();
		if ($last_update == null)
		{
			// nothing to do if disk usage report has not been run
			$this->tpl->setVariable('LAST_UPDATE_TEXT',$lng->txt('disk_quota_report_not_run_yet'));
			return;
		}
		else
		{
			$this->tpl->setVariable('LAST_UPDATE_TEXT',$lng->txt('last_update').': '.ilFormat::formatDate($last_update,'datetime',true));
		}

		// Filter
		$_SESSION['quota_usage_filter'] = isset($_POST['usage_filter']) ? $_POST['usage_filter'] : $_SESSION['quota_usage_filter'];
		if ($_SESSION['quota_usage_filter'] == 0)
		{
			$_SESSION['quota_usage_filter'] = 4;
		}
		$_SESSION['quota_access_filter'] = isset($_POST['access_filter']) ? $_POST['access_filter'] : $_SESSION['quota_access_filter'];
		if ($_SESSION['quota_access_filter'] == 0)
		{
			$_SESSION['quota_access_filter'] = 1;
		}
		$usage_action[1] = $lng->txt('all_users');
		$usage_action[2] = $lng->txt('filter_users_without_disk_usage');
		$usage_action[3] = $lng->txt('filter_users_with_disk_usage');
		$usage_action[4] = $lng->txt('filter_users_with_exceeded_disk_quota');
		$access_action[1] = $lng->txt('all_users');
		$access_action[2] = $lng->txt('filter_users_with_access');
		$access_action[3] = $lng->txt('filter_users_without_access');

		$select_usage_filter = ilUtil::formSelect($_SESSION['quota_usage_filter'],"usage_filter",$usage_action,false,true);
		$select_access_filter = ilUtil::formSelect($_SESSION['quota_access_filter'],"access_filter",$access_action,false,true);

		$this->tpl->setCurrentBlock("filter");
		$this->tpl->setVariable("FILTER_TXT_FILTER",$lng->txt('filter'));
		$this->tpl->setVariable("SELECT_USAGE_FILTER",$select_usage_filter);
		$this->tpl->setVariable("SELECT_ACCESS_FILTER",$select_access_filter);
		$this->tpl->setVariable("FILTER_ACTION",$this->ctrl->getLinkTarget($this, 'viewDiskQuotaReport'));
		$this->tpl->setVariable("FILTER_NAME",'view');
		$this->tpl->setVariable("FILTER_VALUE",$lng->txt('apply_filter'));
		$this->tpl->parseCurrentBlock();

		// load templates for table
	 	$a_tpl = new ilTemplate('tpl.table.html',true,true);
		$a_tpl->addBlockfile("TBL_CONTENT", "tbl_content", "tpl.obj_tbl_rows.html");

		// create table
		require_once './Services/Table/classes/class.ilTableGUI.php';
		$tbl = new ilTableGUI(0, false);

		// title & header columns
		$header_vars = array('login','firstname','lastname','email','access_until','last_login','disk_quota','disk_usage', 'last_reminder');
		$tbl->setHeaderNames(
			array(
				$lng->txt('login'),
				$lng->txt('firstname'),
				$lng->txt('lastname'),
				$lng->txt('email'),
				$lng->txt('access_until'),
				$lng->txt('last_login'),
				$lng->txt('disk_quota'),
				$lng->txt('disk_usage'),
				$lng->txt('last_reminder')
			)
		);
		$tbl->setHeaderVars(
			$header_vars,
			$this->ctrl->getParameterArray($this,'viewDiskQuotaReport',false)
		);

		$tbl->enable("numinfo_header");
		$tbl->setFormName("cmd");
		$tbl->setSelectAllCheckbox("id");

		// sorting 
		$tbl->setOrderColumn($_GET["sort_by"]);
		$tbl->setOrderDirection($_GET["sort_order"]);

		// fetch the data
		$data = ilDiskQuotaChecker::_fetchDiskQuotaReport(
			$_SESSION['quota_usage_filter'],
			$_SESSION['quota_access_filter'],
			$header_vars[$tbl->getOrderColumn()], $tbl->getOrderDirection());

		// paging
		$tbl->setLimit($_GET["limit"]);
		$tbl->setOffset($_GET["offset"]);
		$tbl->setMaxCount(count($data));

		// footer
		$tbl->setFooter("tblfooter",$lng->txt("previous"),$lng->txt("next"));
		
		// render table
		$tbl->setTemplate($a_tpl);

		// render rows
		$count = 0;
		for ($i = $tbl->getOffset(); $i < count($data) && $i < $tbl->getOffset() + $tbl->getLimit(); $i++)
		{
			$row = $data[$i];

			// build columns
			foreach ($header_vars as $key)
			{
				switch ($key)
				{
					case 'login' :
						//build link
						$this->ctrl->setParameterByClass("ilobjusergui", "ref_id", "7");
						$this->ctrl->setParameterByClass("ilobjusergui", "obj_id", $row["usr_id"]);
						$link = $this->ctrl->getLinkTargetByClass("ilobjusergui", "view");
						$tbl_content_cell = '<a href="'.$link.'">'.htmlspecialchars($row[$key]).'</a>';
						break;
					case 'disk_quota' :
						if ($row['role_id'] == SYSTEM_ROLE_ID)
						{
							$tbl_content_cell = "<span class=\"smallgreen\">".$lng->txt('access_unlimited').'</span>';
						}
						else
						{
							$tbl_content_cell = ilFormat::formatSize($row[$key],'short');
						}
						break;
					case 'disk_usage' :
						if ($row['last_update'] == null)
						{
							$tbl_content_cell = $lng->txt('unknown');
						}
						else if ($row['disk_usage'] > $row['disk_quota'])
						{
						 $tbl_content_cell = "<span class=\"smallred\">".ilFormat::formatSize($row[$key],'short').'</span>';
						}
						else
						{
						 $tbl_content_cell = ilFormat::formatSize($row[$key],'short');
						}
						break;
					case 'access_until' :
						if (! $row['active'])
						{
							 $tbl_content_cell = "<span class=\"smallred\">".$lng->txt('inactive').'</span>';
						}
						else if ($row['time_limit_unlimited'])
						{
							$tbl_content_cell = "<span class=\"smallgreen\">".$lng->txt('access_unlimited').'</span>';
						}
						else if ($row['expired'])
						{
							 $tbl_content_cell = "<span class=\"smallred\">".$lng->txt('access_expired').'</span>';
						}
						else
						{
							$tbl_content_cell = ilFormat::formatDate($row[$key]);
						}
						break;
					case 'last_login' :
					case 'last_reminder' :
						if ($row[$key] == null)
						{
							$tbl_content_cell = $lng->txt('no_date');
						}
						else
						{
							$tbl_content_cell = ilFormat::formatDate($row[$key]);
						}
						break;
					default :
						 $tbl_content_cell = htmlspecialchars($row[$key]);
				}
				/*
				if (is_array($tbl_content_cell))
				{
					$tbl->tpl->setCurrentBlock("tbl_cell_subtitle");
					$tbl->tpl->setVariable("TBL_CELL_SUBTITLE",$tbl_content_cell[1]);
					$tbl->tpl->parseCurrentBlock();
					$tbl_content_cell = "<b>".$tbl_content_cell[0]."</b>";
				}*/

				$tbl->tpl->setCurrentBlock("tbl_content_cell");
				$tbl->tpl->setVariable("TBL_CONTENT_CELL",$tbl_content_cell);

				$tbl->tpl->parseCurrentBlock();
			}

			$tbl->tpl->setCurrentBlock("tbl_content_row");
			$rowcolor = ilUtil::switchColor($count,"tblrow1","tblrow2");
			$tbl->tpl->setVariable("ROWCOLOR", $rowcolor);
			$tbl->tpl->parseCurrentBlock();

			$count++;
		}
		$tbl->render();

		// Add table to page
		$this->tpl->setVariable("USER_TABLE",$a_tpl->get());
	}

	/**
	* Edit disk quota settings.
	*/
	public function editDiskQuotaMailTemplate()
	{
		global $rbacsystem, $ilErr, $ilSetting, $tpl, $lng, $ilCtrl;

		if (! $rbacsystem->checkAccess("visible,read",$this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}

		$this->tabs_gui->setTabActive('disk_quota');
		$this->addDiskQuotaSubtabs('disk_quota_reminder_mail');

		$this->tpl->addBlockFile('ADM_CONTENT','adm_content','tpl.disk_quota_reminder_mail.html',
			"Services/WebDAV");
		$this->tpl->setVariable("FORMACTION", $this->ctrl->getFormAction($this));
		$this->tpl->setVariable("IMG_MAIL", ilUtil::getImagePath("icon_mail.svg"));

		$lng->loadLanguageModule("meta");
		$lng->loadLanguageModule("mail");
		$this->tpl->setVariable("TXT_NEW_USER_ACCOUNT_MAIL", $lng->txt("disk_quota_reminder_mail"));
		$this->tpl->setVariable("TXT_NEW_USER_ACCOUNT_MAIL_DESC", $lng->txt("disk_quota_reminder_mail_desc"));

		// placeholder help text
		$this->tpl->setVariable("TXT_USE_PLACEHOLDERS", $lng->txt("mail_nacc_use_placeholder"));
		$this->tpl->setVariable("TXT_MAIL_SALUTATION", $lng->txt("mail_nacc_salutation"));
		$this->tpl->setVariable("TXT_FIRST_NAME", $lng->txt("firstname"));
		$this->tpl->setVariable("TXT_LAST_NAME", $lng->txt("lastname"));
		$this->tpl->setVariable("TXT_EMAIL", $lng->txt("email"));
		$this->tpl->setVariable("TXT_LOGIN", $lng->txt("mail_nacc_login"));
		$this->tpl->setVariable("TXT_DISK_QUOTA", $lng->txt("disk_quota"));
		$this->tpl->setVariable("TXT_DISK_USAGE", $lng->txt("disk_usage"));
		$this->tpl->setVariable("TXT_DISK_USAGE_DETAILS", $lng->txt("disk_usage_details"));
		$this->tpl->setVariable("TXT_ADMIN_MAIL", $lng->txt("mail_nacc_admin_mail"));
		$this->tpl->setVariable("TXT_ILIAS_URL", $lng->txt("mail_nacc_ilias_url"));
		$this->tpl->setVariable("TXT_CLIENT_NAME", $lng->txt("mail_nacc_client_name"));

		$langs = $lng->getInstalledLanguages();
		foreach($langs as $lang_key)
		{
			$amail = $this->disk_quota_obj->_lookupReminderMailTemplate($lang_key);
			$this->tpl->setCurrentBlock("mail_block");
			$add = "";
			if ($lang_key == $lng->getDefaultLanguage())
			{
				$add = " (".$lng->txt("default").")";
			}
			$this->tpl->setVariable("TXT_LANGUAGE",
				$lng->txt("meta_l_".$lang_key).$add);
			$this->tpl->setVariable("TXT_BODY", $lng->txt("message_content"));
			$this->tpl->setVariable("TA_BODY", "body_".$lang_key);
			$this->tpl->setVariable("VAL_BODY",
				ilUtil::prepareFormOutput($amail["body"]));
			$this->tpl->setVariable("TXT_SUBJECT", $lng->txt("subject"));
			$this->tpl->setVariable("INPUT_SUBJECT", "subject_".$lang_key);
			$this->tpl->setVariable("VAL_SUBJECT",
				ilUtil::prepareFormOutput($amail["subject"]));
			$this->tpl->setVariable("TXT_SAL_G", $lng->txt("mail_salutation_general"));
			$this->tpl->setVariable("INPUT_SAL_G", "sal_g_".$lang_key);
			$this->tpl->setVariable("VAL_SAL_G",
				ilUtil::prepareFormOutput($amail["sal_g"]));
			$this->tpl->setVariable("TXT_SAL_M", $lng->txt("mail_salutation_male"));
			$this->tpl->setVariable("INPUT_SAL_M", "sal_m_".$lang_key);
			$this->tpl->setVariable("VAL_SAL_M",
				ilUtil::prepareFormOutput($amail["sal_m"]));
			$this->tpl->setVariable("TXT_SAL_F", $lng->txt("mail_salutation_female"));
			$this->tpl->setVariable("INPUT_SAL_F", "sal_f_".$lang_key);
			$this->tpl->setVariable("VAL_SAL_F",
				ilUtil::prepareFormOutput($amail["sal_f"]));
			$this->tpl->parseCurrentBlock();
		}
		$this->tpl->setVariable("TXT_CANCEL", $lng->txt("cancel"));
		$this->tpl->setVariable("TXT_SAVE", $lng->txt("save"));
	}
	function cancelDiskQuotaMailTemplate()
	{
		$this->ctrl->redirect($this, "editDiskQuotaSettings");
	}

	function saveDiskQuotaMailTemplate()
	{
		global $lng;

		ilUtil::sendSuccess($this->lng->txt("msg_obj_modified"), true);
		$langs = $lng->getInstalledLanguages();
		foreach($langs as $lang_key)
		{
			$this->disk_quota_obj->_writeReminderMailTemplate($lang_key,
				ilUtil::stripSlashes($_POST["subject_".$lang_key]),
				ilUtil::stripSlashes($_POST["sal_g_".$lang_key]),
				ilUtil::stripSlashes($_POST["sal_f_".$lang_key]),
				ilUtil::stripSlashes($_POST["sal_m_".$lang_key]),
				ilUtil::stripSlashes($_POST["body_".$lang_key]));
		}
		$this->ctrl->redirect($this, "editDiskQuotaMailTemplate");
	}
	
	/**
	* Initializes the upload settings form.
	*/
	private function initUploadSettingsForm()
	{
		global $ilCtrl, $lng;

		require_once("./Services/Form/classes/class.ilPropertyFormGUI.php");
		$form = new ilPropertyFormGUI();
		$form->setFormAction($ilCtrl->getFormAction($this));
		$form->setTitle($lng->txt("settings"));
		
		require_once("Services/FileUpload/classes/class.ilFileUploadSettings.php");
		
		// drag and drop file upload
		$chk_enabled = new ilCheckboxInputGUI($lng->txt("enable_dnd_upload"), "enable_dnd_upload");
		$chk_enabled->setValue('1');
		$chk_enabled->setChecked(ilFileUploadSettings::isDragAndDropUploadEnabled());
		$chk_enabled->setInfo($lng->txt('enable_dnd_upload_info'));
		$form->addItem($chk_enabled);
		
		// drag and drop file upload in repository
		$chk_repo = new ilCheckboxInputGUI($lng->txt("enable_repository_dnd_upload"), "enable_repository_dnd_upload");
		$chk_repo->setValue('1');
		$chk_repo->setChecked(ilFileUploadSettings::isRepositoryDragAndDropUploadEnabled());
		$chk_repo->setInfo($lng->txt('enable_repository_dnd_upload_info'));
		$chk_enabled->addSubItem($chk_repo);
		
		// concurrent uploads
		$num_prop = new ilNumberInputGUI($lng->txt("concurrent_uploads"), "concurrent_uploads");
		$num_prop->setDecimals(0);
		$num_prop->setMinValue(1);
		$num_prop->setMinvalueShouldBeGreater(false);
		$num_prop->setMaxValue(ilFileUploadSettings::CONCURRENT_UPLOADS_MAX);
		$num_prop->setMaxvalueShouldBeLess(false);
		$num_prop->setMaxLength(5);
		$num_prop->setSize(10);
		$num_prop->setValue(ilFileUploadSettings::getConcurrentUploads());
		$num_prop->setInfo($lng->txt('concurrent_uploads_info'));
		$chk_enabled->addSubItem($num_prop);
		
		// file suffix replacement
		$ti = new ilTextInputGUI($this->lng->txt("file_suffix_repl"), "suffix_repl_additional");
		$ti->setMaxLength(200);
		$ti->setSize(40);
		$ti->setInfo($this->lng->txt("file_suffix_repl_info")." ".SUFFIX_REPL_DEFAULT);
		$form->addItem($ti);

		// command buttons
		$form->addCommandButton('saveUploadSettings', $lng->txt('save'));
		$form->addCommandButton('view', $lng->txt('cancel'));
		
		return $form;
	}

	/**
	 * Edit upload settings.
	 */
	public function editUploadSettings()
	{
		global $rbacsystem, $ilErr, $tpl, $lng, $ilSetting;

		$this->tabs_gui->setTabActive('upload_settings');

		if (!$rbacsystem->checkAccess("visible,read", $this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}
		
		// get form
		$form = $this->initUploadSettingsForm();
		
		require_once("Services/FileUpload/classes/class.ilFileUploadSettings.php");
		
		// set current values
		$val = array();
		$val["enable_dnd_upload"] = ilFileUploadSettings::isDragAndDropUploadEnabled();
		$val["enable_repository_dnd_upload"] = ilFileUploadSettings::isRepositoryDragAndDropUploadEnabled();
		$val["concurrent_uploads"] = ilFileUploadSettings::getConcurrentUploads();
		$val["suffix_repl_additional"] = $ilSetting->get("suffix_repl_additional");
		$form->setValuesByArray($val);

		// set content
		$tpl->setContent($form->getHTML());
	}

	/**
	 * Save upload settings
	 */
	public function saveUploadSettings()
	{
		global $rbacsystem, $ilErr, $ilCtrl, $lng, $tpl, $ilSetting;

		if (!$rbacsystem->checkAccess("write",$this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}
		
		// get form
		$form = $this->initUploadSettingsForm();
		if ($form->checkInput())
		{
			require_once("Services/FileUpload/classes/class.ilFileUploadSettings.php");
			ilFileUploadSettings::setDragAndDropUploadEnabled($_POST["enable_dnd_upload"] == 1);
			ilFileUploadSettings::setRepositoryDragAndDropUploadEnabled($_POST["enable_repository_dnd_upload"] == 1);
			ilFileUploadSettings::setConcurrentUploads($_POST["concurrent_uploads"]);
			
			// file suffic replacements
			$ilSetting->set("suffix_repl_additional", $_POST["suffix_repl_additional"]);

			ilUtil::sendSuccess($lng->txt('settings_saved'), true);
			$ilCtrl->redirect($this, "editUploadSettings");
		}
		else
		{
			$form->setValuesByPost();	
			$tpl->setContent($form->getHTML());
		}
	}
	
	/**
	* Initializes the preview settings form.
	*/
	private function initPreviewSettingsForm()
	{
		global $ilCtrl, $lng;

		require_once("./Services/Form/classes/class.ilPropertyFormGUI.php");
		$form = new ilPropertyFormGUI();
		$form->setFormAction($ilCtrl->getFormAction($this));
		$form->setTitle($lng->txt("settings"));
		
		require_once("Services/Preview/classes/class.ilPreviewSettings.php");
		
		// drag and drop file upload in repository
		$chk_prop = new ilCheckboxInputGUI($lng->txt("enable_preview"), "enable_preview");
		$chk_prop->setValue('1');
		$chk_prop->setChecked(ilPreviewSettings::isPreviewEnabled());
		$chk_prop->setInfo($lng->txt('enable_preview_info'));
		$form->addItem($chk_prop);
		
		$num_prop = new ilNumberInputGUI($lng->txt("max_previews_per_object"), "max_previews_per_object");
		$num_prop->setDecimals(0);
		$num_prop->setMinValue(1);
		$num_prop->setMinvalueShouldBeGreater(false);
		$num_prop->setMaxValue(ilPreviewSettings::MAX_PREVIEWS_MAX);
		$num_prop->setMaxvalueShouldBeLess(false);
		$num_prop->setMaxLength(5);
		$num_prop->setSize(10);
		$num_prop->setValue(ilPreviewSettings::getMaximumPreviews());
		$num_prop->setInfo($lng->txt('max_previews_per_object_info'));
		$form->addItem($num_prop);

		// command buttons
		$form->addCommandButton('savePreviewSettings', $lng->txt('save'));
		$form->addCommandButton('view', $lng->txt('cancel'));
		
		return $form;
	}

	/**
	 * Edit preview settings.
	 */
	public function editPreviewSettings()
	{
		global $rbacsystem, $ilErr, $tpl, $lng;

		$this->tabs_gui->setTabActive('preview_settings');

		if (!$rbacsystem->checkAccess("visible,read", $this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}
		
		// set warning if ghostscript not installed
		if (!is_file(PATH_TO_GHOSTSCRIPT))
		{
			ilUtil::sendInfo($lng->txt("ghostscript_not_configured"));
		}
		
		// get form
		$form = $this->initPreviewSettingsForm();
		
		// set current values
		require_once("Services/Preview/classes/class.ilPreviewSettings.php");
		
		$val = array();
		$val["enable_preview"] = ilPreviewSettings::isPreviewEnabled();
		$val["max_previews_per_object"] = ilPreviewSettings::getMaximumPreviews();
		$form->setValuesByArray($val);
		
		$html = $form->getHTML();
		
		// build renderer HTML
		require_once("Services/Preview/classes/class.ilRendererFactory.php");
		require_once("Services/Preview/classes/class.ilRendererTableGUI.php");
		
		$renderers = ilRendererFactory::getRenderers();
		
		$table = new ilRendererTableGUI($this, array("editPreviewSettings", "view"));
		$table->setMaxCount(sizeof($renderers));
		$table->setData($renderers);	
		
		$html .= "<br/>" . $table->getHTML();
		
		// set content
		$tpl->setContent($html);
	}

	/**
	 * Save preview settings
	 */
	public function savePreviewSettings()
	{
		global $rbacsystem, $ilErr, $ilCtrl, $tpl, $lng;

		$this->tabs_gui->setTabActive('preview_settings');

		if (!$rbacsystem->checkAccess("write",$this->object->getRefId()))
		{
			$ilErr->raiseError($lng->txt("no_permission"),$ilErr->WARNING);
		}
		
		// get form
		$form = $this->initPreviewSettingsForm();
		if ($form->checkInput())
		{
			require_once("Services/Preview/classes/class.ilPreviewSettings.php");
			ilPreviewSettings::setPreviewEnabled($_POST["enable_preview"] == 1);
			ilPreviewSettings::setMaximumPreviews($_POST["max_previews_per_object"]);

			ilUtil::sendSuccess($lng->txt('settings_saved'), true);
			$ilCtrl->redirect($this, "editPreviewSettings");
		}
		else
		{
			$form->setValuesByPost();	
			$tpl->setContent($form->getHTML());
		}
	}
	
	public function addToExternalSettingsForm($a_form_id)
	{
		global $ilSetting;
		
		switch($a_form_id)
		{
			case ilAdministrationSettingsFormHandler::FORM_SECURITY:
				
				$fields = array('file_suffix_repl' => $ilSetting->get("suffix_repl_additional"));
				
				return array(array("editUploadSettings", $fields));				
		}
	}
} 

?>