<?php

/* Copyright (c) 1998-2010 ILIAS open source, Extended GPL, see docs/LICENSE */

require_once "./Services/Object/classes/class.ilObject2GUI.php";
include_once('./Modules/WebResource/classes/class.ilParameterAppender.php');
require_once 'Services/LinkChecker/interfaces/interface.ilLinkCheckerGUIRowHandling.php';

/**
* Class ilObjLinkResourceGUI
*
* @author Stefan Meyer <smeyer.ilias@gmx.de> 
* @version $Id$
* 
* @ilCtrl_Calls ilObjLinkResourceGUI: ilMDEditorGUI, ilPermissionGUI, ilInfoScreenGUI, ilObjectCopyGUI
* @ilCtrl_Calls ilObjLinkResourceGUI: ilExportGUI, ilWorkspaceAccessGUI, ilCommonActionDispatcherGUI
* @ilCtrl_Calls ilObjLinkResourceGUI: ilPropertyFormGUI, ilInternalLinkGUI
* 
*
* @ingroup ModulesWebResource
*/
class ilObjLinkResourceGUI extends ilObject2GUI implements ilLinkCheckerGUIRowHandling
{
	const VIEW_MODE_VIEW = 1;
	const VIEW_MODE_MANAGE = 2;
	const VIEW_MODE_SORT = 3;
	
	const LINK_MOD_CREATE = 1;
	const LINK_MOD_EDIT = 2;
	const LINK_MOD_ADD = 3;
	
	function getType()
	{
		return "webr";
	}

	public function executeCommand()
	{
		global $ilCtrl, $ilTabs, $ilErr, $ilAccess;
		
		
		//if($this->ctrl->getTargetScript() == 'link_resources.php')
		if($_GET["baseClass"] == 'ilLinkResourceHandlerGUI')
		{
			$_GET['view_mode'] = isset($_GET['switch_mode']) ? $_GET['switch_mode'] : $_GET['view_mode'];
			$ilCtrl->saveParameter($this, 'view_mode');
			$this->__prepareOutput();
		}
			
		$this->lng->loadLanguageModule("webr");

		$next_class = $this->ctrl->getNextClass($this);
		$cmd = $this->ctrl->getCmd();
		
		switch($next_class)
		{
			case "ilinfoscreengui":
				$this->prepareOutput();	
				$this->infoScreenForward();	// forwards command
				break;

			case 'ilmdeditorgui':
				if(!$ilAccess->checkAccess('write','',$this->object->getRefId()))
				{
					$ilErr->raiseError($this->lng->txt('permission_denied'),$ilErr->WARNING);
				}
				
				$this->prepareOutput();	
				$ilTabs->activateTab('id_meta_data');
				include_once 'Services/MetaData/classes/class.ilMDEditorGUI.php';
				$md_gui =& new ilMDEditorGUI($this->object->getId(), 0, $this->object->getType());
				$md_gui->addObserver($this->object,'MDUpdateListener','General');
				$this->ctrl->forwardCommand($md_gui);
				break;
				
			case 'ilpermissiongui':
				$this->prepareOutput();	
				$ilTabs->activateTab('id_permissions');
				include_once("Services/AccessControl/classes/class.ilPermissionGUI.php");
				$perm_gui =& new ilPermissionGUI($this);
				$ret =& $this->ctrl->forwardCommand($perm_gui);
				break;
				
			case 'ilobjectcopygui':
				$this->prepareOutput();	
				include_once './Services/Object/classes/class.ilObjectCopyGUI.php';
				$cp = new ilObjectCopyGUI($this);
				$cp->setType('webr');
				$this->ctrl->forwardCommand($cp);
				break;
				
			case 'ilexportgui':
				$this->prepareOutput();	
				$this->tabs_gui->setTabActive('export');
				include_once './Services/Export/classes/class.ilExportGUI.php';
				$exp = new ilExportGUI($this);
				$exp->addFormat('xml');
				$this->ctrl->forwardCommand($exp);
				break;
			
			case "ilcommonactiondispatchergui":
				include_once("Services/Object/classes/class.ilCommonActionDispatcherGUI.php");
				$gui = ilCommonActionDispatcherGUI::getInstanceFromAjaxCall();
				$this->ctrl->forwardCommand($gui);
				break;
			
			case "ilpropertyformgui":													
				include_once './Services/Form/classes/class.ilPropertyFormGUI.php';
				$this->initFormLink(self::LINK_MOD_EDIT);
				$this->ctrl->forwardCommand($this->form);
				break;
			
			case "ilinternallinkgui":
				$this->lng->loadLanguageModule("content");
				require_once("./Services/Link/classes/class.ilInternalLinkGUI.php");
				$link_gui = new ilInternalLinkGUI("RepositoryItem", 0);
				$link_gui->filterLinkType("PageObject");
				$link_gui->filterLinkType("GlossaryItem");
				$link_gui->filterLinkType("RepositoryItem");
				$link_gui->setFilterWhiteList(true);
				$link_gui->setMode("asynch");			
				$ilCtrl->forwardCommand($link_gui);
				break;
			
			default:
				if(!$cmd)
				{
					$this->ctrl->setCmd("view");
				}
				parent::executeCommand();
		}
		
		if(!$this->getCreationMode())
		{
			// Fill meta header tags
			include_once('Services/MetaData/classes/class.ilMDUtils.php');
			ilMDUtils::_fillHTMLMetaTags($this->object->getId(),$this->object->getId(),'webr');
			
			$this->addHeaderAction();
		}
		return true;
	}
	
	protected function initCreateForm($a_new_type)
	{
		$this->initFormLink(self::LINK_MOD_CREATE);
		return $this->form;
	}

	/**
	 * Save new object
	 * @access	public
	 */
	public function save()
	{
		global $ilCtrl;

		$this->initFormLink(self::LINK_MOD_CREATE);
		if($this->checkLinkInput(self::LINK_MOD_CREATE,0,0))
		{
			// Save new object
			$_POST['title'] = $_POST['tit'];
			$_POST['desc'] = $_POST['des'];
			parent::save();
		}
		else
		{
			// Data incomplete or invalid
			ilUtil::sendFailure($this->lng->txt('err_check_input'));
			$this->form->setValuesByPost();
			$this->tpl->setContent($this->form->getHTML());
		}
	}

	protected function afterSave(ilObject $a_new_object)
	{
		// Save link
		$this->link->setLinkResourceId($a_new_object->getId());
		$link_id = $this->link->add();
		$this->link->updateValid(true);

		ilUtil::sendSuccess($this->lng->txt('webr_link_added'));
		
		// personal workspace
		if($this->id_type == self::WORKSPACE_NODE_ID)
		{
			$this->ctrl->redirect($this, "editLinks");			
		}
		// repository
		else
		{
			ilUtil::redirect("ilias.php?baseClass=ilLinkResourceHandlerGUI&ref_id=".
				$a_new_object->getRefId()."&cmd=switchViewMode&switch_mode=2");
		}
	}
	
	/**
	 * Edit settings
	 * Title, Description, Sorting
	 * @return 
	 */
	protected function settings()
	{
		global $ilTabs;
	
		$this->checkPermission('write');
		$ilTabs->activateTab('id_settings');
		
		$this->initFormSettings();
		$this->tpl->setContent($this->form->getHTML());
	}
	
	/**
	 * Save container settings
	 * @return 
	 */
	protected function saveSettings()
	{
		global $ilTabs;
		
		$this->checkPermission('write');
		$ilTabs->activateTab('id_settings');
		
		$this->initFormSettings();
		if($this->form->checkInput())
		{
			$this->object->setTitle($this->form->getInput('tit'));
			$this->object->setDescription($this->form->getInput('des'));
			$this->object->update();
			
			include_once './Services/Container/classes/class.ilContainerSortingSettings.php';
			$sort = new ilContainerSortingSettings($this->object->getId());
			$sort->setSortMode($this->form->getInput('sor'));
			$sort->update();
			
			ilUtil::sendSuccess($this->lng->txt('settings_saved'),true);
			$this->ctrl->redirect($this,'settings');
		}
		
		$this->form->setValuesByPost();
		ilUtil::sendFailure($this->lng->txt('err_check_input'));
		$this->tpl->setContent($this->form->getHTML());
	}
	
	
	/**
	 * Show settings form
	 * @return 
	 */
	protected function initFormSettings()
	{
		include_once './Services/Form/classes/class.ilPropertyFormGUI.php';
		$this->form = new ilPropertyFormGUI();
		$this->form->setFormAction($this->ctrl->getFormAction($this,'saveSettings'));
		$this->form->setTitle($this->lng->txt('webr_edit_settings'));
		
		// Title
		$tit = new ilTextInputGUI($this->lng->txt('webr_list_title'),'tit');
		$tit->setValue($this->object->getTitle());
		$tit->setRequired(true);
		$tit->setSize(40);
		$tit->setMaxLength(127);
		$this->form->addItem($tit);
		
		// Description
		$des = new ilTextAreaInputGUI($this->lng->txt('webr_list_desc'),'des');
		$des->setValue($this->object->getDescription());
		$des->setCols(40);
		$des->setRows(3);
		$this->form->addItem($des);
		
		// Sorting
		include_once './Services/Container/classes/class.ilContainerSortingSettings.php';
		include_once './Services/Container/classes/class.ilContainer.php';
		
		$sor = new ilRadioGroupInputGUI($this->lng->txt('webr_sorting'),'sor');
		$sor->setRequired(true);
		include_once './Services/Container/classes/class.ilContainerSortingSettings.php';
		$sor->setValue(ilContainerSortingSettings::_lookupSortMode($this->object->getId()));
		
		$opt = new ilRadioOption(
			$this->lng->txt('webr_sort_title'),
			ilContainer::SORT_TITLE
		);
		$sor->addOption($opt);
		
		$opm = new ilRadioOption(
			$this->lng->txt('webr_sort_manual'),
			ilContainer::SORT_MANUAL
		);
		$sor->addOption($opm);
		$this->form->addItem($sor);

		$this->form->addCommandButton('saveSettings', $this->lng->txt('save'));
		$this->form->addCommandButton('view', $this->lng->txt('cancel'));
	}
	

	/**
	 * Edit a single link
	 * @return 
	 */
	public function editLink()
	{
		global $ilCtrl;
		
		$this->checkPermission('write');
		$this->activateTabs('content','id_content_view');
		
		if(!(int) $_GET['link_id'])
		{
			ilUtil::sendFailure($this->lng->txt('select_one'),true);
			$ilCtrl->redirect($this,'view');
		}
		
		$this->initFormLink(self::LINK_MOD_EDIT);
		$this->setValuesFromLink((int) $_GET['link_id']);
		$this->tpl->setContent($this->form->getHTML());
	}
	
	/**
	 * Save after editing
	 * @return 
	 */
	public function updateLink()
	{
		global $ilCtrl;

		$this->initFormLink(self::LINK_MOD_EDIT);
		if($this->checkLinkInput(self::LINK_MOD_EDIT,$this->object->getId(),(int) $_REQUEST['link_id']))
		{
			$this->link->setLinkId((int) $_REQUEST['link_id']);
			$this->link->update();
			if(ilParameterAppender::_isEnabled() and is_object($this->dynamic))
			{
				$this->dynamic->add((int) $_REQUEST['link_id']);
			}
			
			if($this->isContainerMetaDataRequired())
			{
				$this->object->setTitle($this->form->getInput('tit'));
				$this->object->setDescription($this->form->getInput('des'));
				$this->object->update();
			}
			
			ilUtil::sendSuccess($this->lng->txt('settings_saved'),true);
			$ilCtrl->redirect($this,'view');
		}
		ilUtil::sendFailure($this->lng->txt('err_check_input'));
		$this->form->setValuesByPost();
		$this->tpl->setContent($this->form->getHTML());
	}
	
	/**
	 * Add an additional link
	 * @return 
	 */
	public function addLink()
	{
		$this->checkPermission('write');
		$this->activateTabs('content','id_content_view');
	
		$this->initFormLink(self::LINK_MOD_ADD);
		$this->tpl->setContent($this->form->getHTML());
	}
	
	/**
	 * Save form data
	 * @return 
	 */
	public function saveAddLink()
	{
		global $ilCtrl;
		
		$this->checkPermission('write');
	
		$this->initFormLink(self::LINK_MOD_ADD);
		if($this->checkLinkInput(self::LINK_MOD_ADD,$this->object->getId(),0))
		{
			if($this->isContainerMetaDataRequired())
			{
				// Save list data
				$this->object->setTitle($this->form->getInput('lti'));
				$this->object->setDescription($this->form->getInput('tde'));
				$this->object->update();
			}
			
			// Save Link
			$link_id = $this->link->add();
			$this->link->updateValid(true);
			
			// Dynamic parameters
			if(ilParameterAppender::_isEnabled() and is_object($this->dynamic))
			{
				$this->dynamic->add($link_id);
			}
			ilUtil::sendSuccess($this->lng->txt('webr_link_added'),true);
			$ilCtrl->redirect($this,'view');
		}
		// Error handling
		ilUtil::sendFailure($this->lng->txt('err_check_input'));
		$this->form->setValuesByPost();
		
		$this->activateTabs('content','id_content_view');
		$this->tpl->setContent($this->form->getHTML());
	}
	
	/**
	 * Delete a dynamic parameter
	 * @return 
	 */
	protected function deleteParameter()
	{
		global $ilCtrl;

		$this->checkPermission('write');
		
		$this->ctrl->setParameter($this,'link_id',(int) $_GET['link_id']);
		
		if(!isset($_GET['param_id']))
		{
			ilUtil::sendFailure($this->lng->txt('select_one'),TRUE);
			$ilCtrl->redirect($this,'view');
		}	

		include_once './Modules/WebResource/classes/class.ilParameterAppender.php';
		$param = new ilParameterAppender($this->object->getId());
		$param->delete((int) $_GET['param_id']);
		
		ilUtil::sendSuccess($this->lng->txt('links_parameter_deleted'),true);
		$ilCtrl->redirect($this,'editLinks');
	}
	
	protected function deleteParameterForm()
	{
		global $ilCtrl;

		$this->checkPermission('write');
		
		if(!isset($_GET['param_id']))
		{
			ilUtil::sendFailure($this->lng->txt('select_one'),TRUE);
			$ilCtrl->redirect($this,'view');
		}	

		include_once './Modules/WebResource/classes/class.ilParameterAppender.php';
		$param = new ilParameterAppender($this->object->getId());
		$param->delete((int) $_GET['param_id']);
		
		ilUtil::sendSuccess($this->lng->txt('links_parameter_deleted'),true);
		$ilCtrl->redirect($this,'view');
		
	}
	
	
	/**
	 * Update all visible links
	 * @return 
	 */
	protected function updateLinks()
	{
		global $ilCtrl;
		
		$this->checkPermission('write');
		$this->activateTabs('content','');
		
		if(!is_array($_POST['ids']))
		{
			ilUtil::sendFailure($this->lng->txt('select_one'),TRUE);
			$ilCtrl->redirect($this,'view');
		}
		
		// Validate
		$invalid = array();
		foreach($_POST['ids'] as $link_id)
		{
			$data = $_POST['links'][$link_id];
	
			// handle internal links
			if($_POST['tar_'.$link_id.'_ajax_type'] &&
				$_POST['tar_'.$link_id.'_ajax_id'])
			{
				$data['tar'] = $_POST['links'][$link_id]['tar'] = 
					$_POST['tar_'.$link_id.'_ajax_type'].'|'.
					$_POST['tar_'.$link_id.'_ajax_id'];	
			}
			
			
			if(!strlen($data['tit']))
			{
				$invalid[] = $link_id;
				continue;
			}
			if(!strlen($data['tar']))
			{
				$invalid[] = $link_id;
				continue;
			}
			if($data['nam'] and !$data['val'])
			{
				$invalid[] = $link_id;
				continue;
			}
			if(!$data['nam'] and $data['val'])
			{
				$invalid[] = $link_id;
				continue;
			}
		}
		
		if(count($invalid))
		{
			ilUtil::sendFailure($this->lng->txt('err_check_input'));
			$this->tpl->addBlockFile('ADM_CONTENT','adm_content','tpl.webr_manage.html','Modules/WebResource');
			
			include_once './Modules/WebResource/classes/class.ilWebResourceEditableLinkTableGUI.php';
			$table = new ilWebResourceEditableLinkTableGUI($this,'view');
			$table->setInvalidLinks($invalid);
			$table->parseSelectedLinks($_POST['ids']);
			$table->updateFromPost();
			$this->tpl->setVariable('TABLE_LINKS',$table->getHTML());
			return false;
		}
		
		include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
		$links = new ilLinkResourceItems($this->object->getId());
		
		// Save Settings
		foreach($_POST['ids'] as $link_id)
		{
			$data = $_POST['links'][$link_id];
			
			$orig = ilLinkResourceItems::lookupItem($this->object->getId(),$link_id);
			
			$links->setLinkId($link_id);
			$links->setTitle(ilUtil::stripSlashes($data['tit']));
			$links->setDescription(ilUtil::stripSlashes($data['des']));
			$links->setTarget(str_replace('"', '', ilUtil::stripSlashes($data['tar'])));
			$links->setActiveStatus((int) $data['act']);
			$links->setDisableCheckStatus((int) $data['che']);
			$links->setLastCheckDate($orig['last_check']);
			$links->setValidStatus((int) $data['vali']);
			$links->update();
			
			if(strlen($data['nam']) and $data['val'])
			{
				$param = new ilParameterAppender($this->object->getId());
				$param->setName(ilUtil::stripSlashes($data['nam']));
				$param->setValue((int) $data['val']);
				$param->add($link_id);
			}

			if($this->isContainerMetaDataRequired())
			{
				$this->object->setTitle(ilUtil::stripSlashes($data['tit']));
				$this->object->setDescription(ilUtil::stripSlashes($data['des']));
				$this->object->update();
			}
			
			// TODO: Dynamic parameters
		}
			
		ilUtil::sendSuccess($this->lng->txt('settings_saved'),TRUE);
		$ilCtrl->redirect($this,'view');							
	}
	
	/**
	 * Set form values from link
	 * @param object $a_link_id
	 * @return 
	 */
	protected function setValuesFromLink($a_link_id)
	{
		include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
		$link = new ilLinkResourceItems($this->object->getId());
		
		$values = $link->getItem($a_link_id);
		
		if(ilParameterAppender::_isEnabled())
		{
		}
		
		$this->form->setValuesByArray(
			array(
				'tit'		=> $values['title'],
				'tar'		=> $values['target'],
				'des'		=> $values['description'],
				'act'		=> (int) $values['active'],
				'che'		=> (int) $values['disable_check'],
				'vali'		=> (int) $values['valid']
			)
		);				
	}
	
	
	/**
	 * Check input after creating a new link
	 * @param object $a_mode
	 * @param object $a_webr_id [optional]
	 * @param object $a_link_id [optional]
	 * @return 
	 */
	protected function checkLinkInput($a_mode,$a_webr_id = 0,$a_link_id = 0)
	{
		$valid = $this->form->checkInput();
		
		include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
		$this->link = new ilLinkResourceItems($a_webr_id);
		$this->link->setTarget(str_replace('"', '', ilUtil::stripSlashes($this->form->getInput('tar'))));
		$this->link->setTitle($this->form->getInput('tit'));
		$this->link->setDescription($this->form->getInput('des'));
		$this->link->setDisableCheckStatus($this->form->getInput('che'));
		
		if($a_mode == self::LINK_MOD_CREATE)
		{
			$this->link->setActiveStatus(true);
		}
		else
		{
			$this->link->setActiveStatus($this->form->getInput('act'));
		}
		
		if($a_mode == self::LINK_MOD_EDIT)
		{
			$this->link->setValidStatus($this->form->getInput('vali'));
		}
		else
		{
			$this->link->setValidStatus(true);
		}
		
		if(!ilParameterAppender::_isEnabled())
		{
			return $valid;
		}
		
		$this->dynamic = new ilParameterAppender($a_webr_id);
		$this->dynamic->setName($this->form->getInput('nam'));
		$this->dynamic->setValue($this->form->getInput('val'));
		if(!$this->dynamic->validate())
		{
			switch($this->dynamic->getErrorCode())
			{
				case LINKS_ERR_NO_NAME:
					$this->form->getItemByPostVar('nam')->setAlert($this->lng->txt('links_no_name_given'));
					return false;
					
				case LINKS_ERR_NO_VALUE:
					$this->form->getItemByPostVar('val')->setAlert($this->lng->txt('links_no_value_given'));
					return false;
					
				case LINKS_ERR_NO_NAME_VALUE:
					// Nothing entered => no error
					return $valid;
			}
			$this->dynamic = null;
		}
		return $valid;
	}

	
	/**
	 * Show create/edit single link
	 * @param int form mode
	 * @return 
	 */
	protected function initFormLink($a_mode)
	{
		global $ilTabs;
	
		$ilTabs->activateTab("id_content");
		
		include_once './Services/Form/classes/class.ilPropertyFormGUI.php';
		$this->form = new ilPropertyFormGUI();
		
		switch($a_mode)
		{
			case self::LINK_MOD_CREATE:
				// Header
				$this->ctrl->setParameter($this,'new_type','webr');
				$this->form->setTitle($this->lng->txt('webr_new_link'));
				$this->form->setTableWidth('600px');

				// Buttons
				$this->form->addCommandButton('save', $this->lng->txt('webr_add'));
				$this->form->addCommandButton('cancel', $this->lng->txt('cancel'));
				break;
				
			case self::LINK_MOD_ADD:
				// Header
				$this->form->setTitle($this->lng->txt('webr_new_link'));

				// Buttons
				$this->form->addCommandButton('saveAddLink', $this->lng->txt('webr_add'));
				$this->form->addCommandButton('view', $this->lng->txt('cancel'));
				break;

			case self::LINK_MOD_EDIT:
				// Header
				$this->ctrl->setParameter($this,'link_id',(int) $_REQUEST['link_id']);
				$this->form->setTitle($this->lng->txt('webr_edit'));
				
				// Buttons
				$this->form->addCommandButton('updateLink', $this->lng->txt('save'));
				$this->form->addCommandButton('view', $this->lng->txt('cancel'));
				break;			
		}
		
		
		$this->form->setFormAction($this->ctrl->getFormAction($this));
		
		if($a_mode == self::LINK_MOD_ADD and $this->isContainerMetaDataRequired())
		{
			ilUtil::sendInfo($this->lng->txt('webr_container_info'));
			
			
			$this->form->setTitle($this->lng->txt('webr_edit_list'));
			
			// List Title
			$title = new ilTextInputGUI($this->lng->txt('webr_list_title'),'lti');
			$title->setRequired(true);
			$title->setSize(40);
			$title->setMaxLength(127);
			$this->form->addItem($title);
			
			// List Description
			$desc = new ilTextAreaInputGUI($this->lng->txt('webr_list_desc'),'tde');
			$desc->setRows(3);
			$desc->setCols(40);
			$this->form->addItem($desc);
			
			// Addtional section
			$sect = new ilFormSectionHeaderGUI();
			$sect->setTitle($this->lng->txt('webr_add'));
			$this->form->addItem($sect);
		}

		// Target
		/*
		$tar = new ilTextInputGUI($this->lng->txt('webr_link_target'),'tar');
		$tar->setValue("http://");
		
		$tar->setSize(40);
		$tar->setMaxLength(500);
		*/
		include_once 'Services/Form/classes/class.ilLinkInputGUI.php';
		$tar = new ilLinkInputGUI($this->lng->txt('webr_link_target'),'tar');
		$tar->setInternalLinkFilterTypes(array("PageObject", "GlossaryItem", "RepositoryItem"));
		
		$tar->setRequired(true);
		$this->form->addItem($tar);		
		
		// Title
		$tit = new ilTextInputGUI($this->lng->txt('webr_link_title'),'tit');
		$tit->setRequired(true);
		$tit->setSize(40);
		$tit->setMaxLength(127);
		$this->form->addItem($tit);
		
		// Description
		$des = new ilTextAreaInputGUI($this->lng->txt('description'),'des');
		$des->setRows(3);
		$des->setCols(40);
		$this->form->addItem($des);
		
		
		if($a_mode != self::LINK_MOD_CREATE)
		{
			// Active
			$act = new ilCheckboxInputGUI($this->lng->txt('active'),'act');
			$act->setChecked(true);
			$act->setValue(1);
			$this->form->addItem($act);

			// Check
			$che = new ilCheckboxInputGUI($this->lng->txt('webr_disable_check'),'che');
			$che->setValue(1);
			$this->form->addItem($che);
		}
		
		// Valid
		if($a_mode == self::LINK_MOD_EDIT)
		{
			$val = new ilCheckboxInputGUI($this->lng->txt('valid'),'vali');
			$this->form->addItem($val);
		}
		
		if(ilParameterAppender::_isEnabled() && $a_mode != self::LINK_MOD_CREATE)
		{
			$dyn = new ilNonEditableValueGUI($this->lng->txt('links_dyn_parameter'));
			$dyn->setInfo($this->lng->txt('links_dynamic_info'));
			
			
			if(count($links = ilParameterAppender::_getParams((int) $_GET['link_id'])))
			{
				$ex = new ilCustomInputGUI($this->lng->txt('links_existing_params'),'ex');
				$dyn->addSubItem($ex);
				
				foreach($links as $id => $link)
				{
					$p = new ilCustomInputGUI();
					
					$ptpl = new ilTemplate('tpl.link_dyn_param_edit.html',true,true,'Modules/WebResource');
					$ptpl->setVariable('INFO_TXT',ilParameterAppender::parameterToInfo($link['name'], $link['value']));
					$this->ctrl->setParameter($this,'param_id',$id);
					$ptpl->setVariable('LINK_DEL',$this->ctrl->getLinkTarget($this,'deleteParameterForm'));
					$ptpl->setVariable('LINK_TXT',$this->lng->txt('delete'));
					$p->setHtml($ptpl->get());
					$dyn->addSubItem($p);
				}
			}
			
			// Existing parameters
			
			// New parameter
			if($a_mode != self::LINK_MOD_CREATE)
			{
				#$new = new ilCustomInputGUI($this->lng->txt('links_add_param'),'');
				#$dyn->addSubItem($new);
			}
			
			// Dynyamic name
			$nam = new ilTextInputGUI($this->lng->txt('links_name'),'nam');
			$nam->setSize(12);
			$nam->setMaxLength(128);
			$dyn->addSubItem($nam);
			
			// Dynamic value
			$val = new ilSelectInputGUI($this->lng->txt('links_value'),'val');
			$val->setOptions(ilParameterAppender::_getOptionSelect());
			$val->setValue(0);
			$dyn->addSubItem($val);
			
			$this->form->addItem($dyn);
		}
	}
	
	/**
	 * Check if a new container title is required
	 * Necessary if there is more than one link
	 * @return 
	 */
	protected function isContainerMetaDataRequired()
	{
		include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
		return ilLinkResourceItems::lookupNumberOfLinks($this->object->getId()) == 1;
	}
	
	/**
	 * Switch between "View" "Manage" and "Sort"
	 * @return 
	 */
	protected function switchViewMode()
	{
		global $ilCtrl;
		
		$_REQUEST['view_mode'] = $_GET['view_mode'] = (int) $_GET['switch_mode'];
		$this->view();
	}
	
	/**
	 * Start with manage mode
	 * @return 
	 */
	protected function editLinks()
	{
		$_GET['switch_mode'] = self::VIEW_MODE_MANAGE;
		$this->switchViewMode();
	}
	

	/**
	 * View object 
	 * @return 
	 */
	public function view()
	{
		global $ilErr, $ilTabs;
		
		$ilTabs->activateTab("id_content");
		
		$this->checkPermission('read');
		
		if (strtolower($_GET["baseClass"]) == "iladministrationgui")
		{
			parent::view();
			return true;
		}
		else
		{
			switch((int) $_REQUEST['view_mode'])
			{
				case self::VIEW_MODE_MANAGE:
					$this->manage();
					break;
					
				case self::VIEW_MODE_SORT:
					// #14638
					include_once './Services/Container/classes/class.ilContainerSortingSettings.php';					
					if(ilContainerSortingSettings::_lookupSortMode($this->object->getId()) == ilContainer::SORT_MANUAL)
					{
						$this->sort();
						break;
					}
					// fallthrough
				
				default:
					$this->showLinks();
					break;
			}
		}
		$GLOBALS['tpl']->setPermanentLink($this->object->getType(),$this->object->getRefId());
		return true;
	}
	
	/**
	 * Manage links
	 * @return 
	 */
	protected function manage()
	{						
		$this->checkPermission('write');
		$this->activateTabs('content','id_content_manage');
		
		$this->tpl->addBlockFile('ADM_CONTENT','adm_content','tpl.webr_manage.html','Modules/WebResource');
		$this->showToolbar('ACTION_BUTTONS');
		
		include_once './Modules/WebResource/classes/class.ilWebResourceEditableLinkTableGUI.php';
		$table = new ilWebResourceEditableLinkTableGUI($this,'view');
		$table->parse();
		
		include_once './Services/Link/classes/class.ilInternalLinkGUI.php';
		$js = ilInternalLinkGUI::getInitHTML("");
		
		$this->tpl->addJavaScript("Modules/WebResource/js/intLink.js");
		$this->tpl->addJavascript("Services/Form/js/Form.js");

		$this->tpl->setVariable('TABLE_LINKS',$table->getHTML().$js);
	}
	
	/**
	 * Show all active links
	 * @return 
	 */
	protected function showLinks()
	{
		$this->checkPermission('read');
		$this->activateTabs('content','id_content_view');
		
		include_once './Modules/WebResource/classes/class.ilWebResourceLinkTableGUI.php';
		$table = new ilWebResourceLinkTableGUI($this,'showLinks');
		$table->parse();
		
		$this->tpl->addBlockFile('ADM_CONTENT','adm_content','tpl.webr_view.html','Modules/WebResource');
		$this->showToolbar('ACTION_BUTTONS');
		$this->tpl->setVariable('LINK_TABLE',$table->getHTML());
	}
	
	/**
	 * Sort web links
	 * @return 
	 */
	protected function sort()
	{
		$this->checkPermission('write');
		$this->activateTabs('content','id_content_ordering');
		
		include_once './Modules/WebResource/classes/class.ilWebResourceLinkTableGUI.php';
		$table = new ilWebResourceLinkTableGUI($this,'sort',true);
		$table->parse();
		
		$this->tpl->addBlockFile('ADM_CONTENT','adm_content','tpl.webr_view.html','Modules/WebResource');
		$this->showToolbar('ACTION_BUTTONS');
		$this->tpl->setVariable('LINK_TABLE',$table->getHTML());
	}
	
	/**
	 * Save nmanual sorting
	 * @return 
	 */
	protected function saveSorting()
	{
		$this->checkPermission('write');
		
		include_once './Services/Container/classes/class.ilContainerSorting.php';
		$sort = ilContainerSorting::_getInstance($this->object->getId());
		$sort->savePost((array) $_POST['position']);
		
		ilUtil::sendSuccess($this->lng->txt('settings_saved'),true);
		$this->view();
	}
	
	
	/**
	 * Show toolbar
	 * @param string $a_tpl_var Name of template variable
	 * @return 
	 */
	protected function showToolbar($a_tpl_var)
	{
		if(!$this->checkPermissionBool('write'))
		{
			return;
		}
		
		include_once './Services/UIComponent/Toolbar/classes/class.ilToolbarGUI.php';
		$tool = new ilToolbarGUI();
		$tool->setFormAction($this->ctrl->getFormAction($this));
		$tool->addButton(
			$this->lng->txt('webr_add'),
			$this->ctrl->getLinkTarget($this,'addLink')
		);
		
		$this->tpl->setVariable($a_tpl_var,$tool->getHTML());
		return;
	}
	
	/**
	 * Show delete confirmation screen 
	 * @return 
	 */
	protected function confirmDeleteLink()
	{
		$this->checkPermission('write');
		$this->activateTabs('content','id_content_view');
		
		$link_ids = array();

		if(is_array($_POST['link_ids']))
		{
			$link_ids =$_POST['link_ids'];
		}
		elseif(isset($_GET['link_id']))
		{
			$link_ids = array($_GET['link_id']);
		}

		if(!count($link_ids) > 0)
		{
			ilUtil::sendFailure($this->lng->txt('select_one'));
			$this->view();
			return false;
		}
		
		include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
		$links = new ilLinkResourceItems($this->object->getId());
		
		include_once 'Services/Utilities/classes/class.ilConfirmationGUI.php';
		$confirm = new ilConfirmationGUI();
		$confirm->setFormAction($this->ctrl->getFormAction($this,'view'));
		$confirm->setHeaderText($this->lng->txt('webr_sure_delete_items'));
		$confirm->setConfirm($this->lng->txt('delete'), 'deleteLinks');
		$confirm->setCancel($this->lng->txt('cancel'), 'view');
		
		foreach($link_ids as $link_id)
		{
			$link = $links->getItem($link_id);
			$confirm->addItem('link_ids[]', $link_id,$link['title']);
		}
		$this->tpl->setContent($confirm->getHTML());
	}
	
	/**
	 * Delete links
	 * @return 
	 */
	protected function deleteLinks()
	{
		global $ilCtrl;
		
		$this->checkPermission('write');
		
		include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
		$links = new ilLinkResourceItems($this->object->getId());
		
		foreach($_POST['link_ids'] as $link_id)
		{
			$links->delete($link_id);
		}
		ilUtil::sendSuccess($this->lng->txt('webr_deleted_items'),true);
		$ilCtrl->redirect($this,'view');
	}
	
	/**
	 * Deactivate links
	 * @return 
	 */
	protected function deactivateLink()
	{
		global $ilCtrl;
		
		$this->checkPermission('write');
		
		include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
		$links = new ilLinkResourceItems($this->object->getId());

		if(!$_GET['link_id'])
		{
			ilUtil::sendFailure($this->lng->txt('select_one'),true);
			$ilCtrl->redirect($this,'view');
		}
		
		$links->setLinkId((int) $_GET['link_id']);
		$links->updateActive(false);
		
		ilUtil::sendSuccess($this->lng->txt('webr_inactive_success'),true);
		$ilCtrl->redirect($this,'view');
	}
	

	/**
	* this one is called from the info button in the repository
	* not very nice to set cmdClass/Cmd manually, if everything
	* works through ilCtrl in the future this may be changed
	*/
	function infoScreen()
	{
		$this->ctrl->setCmd("showSummary");
		$this->ctrl->setCmdClass("ilinfoscreengui");
		$this->infoScreenForward();
	}

	/**
	* show information screen
	*/
	function infoScreenForward()
	{
		global $ilTabs;

		$this->checkPermission('visible');
		$ilTabs->activateTab('id_info');

		include_once("./Services/InfoScreen/classes/class.ilInfoScreenGUI.php");
		$info = new ilInfoScreenGUI($this);
		
		$info->enablePrivateNotes();
		
		// standard meta data
		$info->addMetaDataSections($this->object->getId(),0, $this->object->getType());
		
		if($this->id_type == self::WORKSPACE_NODE_ID)
		{			
			$info->addProperty($this->lng->txt("perma_link"), $this->getPermanentLinkWidget());
		}
		
		// forward the command
		$this->ctrl->forwardCommand($info);
	}


	function history()
	{
		global $ilTabs;
		
		$this->checkPermission('write');
		$ilTabs->activateTab('id_history');

		include_once("./Services/History/classes/class.ilHistoryTableGUI.php");
		$hist_gui = new ilHistoryTableGUI($this, "history", $this->object->getId(), $this->object->getType);
		$hist_gui->initTable();
		$this->tpl->setContent($hist_gui->getHTML());
	}
	
	/**
	 * 
	 * @see		ilLinkCheckerGUIRowHandling::formatInvalidLinkArray()
	 * @param	array Unformatted array
	 * @return	array Formatted array
	 * @access	public
	 * 
	 */
	public function formatInvalidLinkArray(Array $row)
	{
		$this->object->items_obj->readItem($row['page_id']);
		$row['title'] = $this->object->items_obj->getTitle();	
	
		require_once 'Services/UIComponent/AdvancedSelectionList/classes/class.ilAdvancedSelectionListGUI.php';
		$actions = new ilAdvancedSelectionListGUI();
		$actions->setSelectionHeaderClass('small');	
		$actions->setItemLinkClass('xsmall');		
		$actions->setListTitle($this->lng->txt('actions'));		
		$actions->setId($row['page_id']);
		$this->ctrl->setParameter($this, 'link_id', $row['page_id']);		
		$actions->addItem(
			$this->lng->txt('edit'),
			'',
			$this->ctrl->getLinkTarget($this, 'editLink')
		);
		$this->ctrl->clearParameters($this);
		$row['action_html'] = $actions->getHTML();		
		
		return $row;
	}

	/**
	 * Show link validation
	 * @return 
	 */
	protected function linkChecker()
	{
		global $ilias, $ilUser, $tpl, $ilTabs;
		
		$this->checkPermission('write');
		$ilTabs->activateTab('id_link_check');

		$this->__initLinkChecker();
		$this->object->initLinkResourceItemsObject();
		
		require_once './Services/LinkChecker/classes/class.ilLinkCheckerTableGUI.php';
		
		$toolbar = new ilToolbarGUI();

		if((bool)$ilias->getSetting('cron_web_resource_check'))
		{
			include_once './Services/LinkChecker/classes/class.ilLinkCheckNotify.php';
			include_once 'Services/Form/classes/class.ilPropertyFormGUI.php';
			
			$chb = new ilCheckboxInputGUI($this->lng->txt('link_check_message_a'), 'link_check_message');
			$chb->setValue(1);
			$chb->setChecked((bool)ilLinkCheckNotify::_getNotifyStatus($ilUser->getId(), $this->object->getId()));
			$chb->setOptionTitle($this->lng->txt('link_check_message_b'));
			
			$toolbar->addInputItem($chb);
			$toolbar->addFormButton($this->lng->txt('save'), 'saveLinkCheck');
			$toolbar->setFormAction($this->ctrl->getLinkTarget($this, 'saveLinkCheck'));
		}

		$tgui = new ilLinkCheckerTableGUI($this, 'linkChecker');
		$tgui->setLinkChecker($this->link_checker_obj)
			 ->setRowHandler($this)
			 ->setRefreshButton($this->lng->txt('refresh'), 'refreshLinkCheck');

		return $tpl->setContent($tgui->prepareHTML()->getHTML().$toolbar->getHTML());
	}
	
	function saveLinkCheck()
	{
		global $ilDB,$ilUser;

		include_once './Services/LinkChecker/classes/class.ilLinkCheckNotify.php';

		$link_check_notify =& new ilLinkCheckNotify($ilDB);
		$link_check_notify->setUserId($ilUser->getId());
		$link_check_notify->setObjId($this->object->getId());

		if($_POST['link_check_message'])
		{
			ilUtil::sendSuccess($this->lng->txt('link_check_message_enabled'));
			$link_check_notify->addNotifier();
		}
		else
		{
			ilUtil::sendSuccess($this->lng->txt('link_check_message_disabled'));
			$link_check_notify->deleteNotifier();
		}
		$this->linkChecker();

		return true;
	}
		


	function refreshLinkCheck()
	{
		$this->__initLinkChecker();

		if(!$this->link_checker_obj->checkPear())
		{
			ilUtil::sendFailure($this->lng->txt('missing_pear_library'));
			$this->linkChecker();

			return false;
		}


		$this->object->initLinkResourceItemsObject();

		// Set all link to valid. After check invalid links will be set to invalid
		$this->object->items_obj->updateValidByCheck();
 		
		foreach($this->link_checker_obj->checkWebResourceLinks() as $invalid)
		{
			$this->object->items_obj->readItem($invalid['page_id']);
			$this->object->items_obj->setActiveStatus(false);
			$this->object->items_obj->setValidStatus(false);
			$this->object->items_obj->update(false);
		}
		
		$this->object->items_obj->updateLastCheck();
		ilUtil::sendSuccess($this->lng->txt('link_checker_refreshed'));

		$this->linkChecker();

		return true;
	}

	function __initLinkChecker()
	{
		global $ilDB;

		include_once './Services/LinkChecker/classes/class.ilLinkChecker.php';

		$this->link_checker_obj =& new ilLinkChecker($ilDB,false);
		$this->link_checker_obj->setObjId($this->object->getId());

		return true;
	}
	
	
	/**
	 * Activate tab and subtabs
	 * @param string $a_active_tab
	 * @param string $a_active_subtab [optional]
	 * @return 
	 */
	protected function activateTabs($a_active_tab,$a_active_subtab = '')
	{
		global $ilCtrl, $ilTabs, $lng;
		
		switch($a_active_tab)
		{
			case 'content':
				if($this->checkPermissionBool('write'))
				{
					$this->lng->loadLanguageModule('cntr');
					
					$this->ctrl->setParameter($this,'switch_mode',self::VIEW_MODE_VIEW);
					$ilTabs->addSubTab('id_content_view',
						$lng->txt('view'),
						$this->ctrl->getLinkTarget($this,'switchViewMode')
					);
					$this->ctrl->setParameter($this,'switch_mode',self::VIEW_MODE_MANAGE);
					$ilTabs->addSubTab('id_content_manage',
						$lng->txt('cntr_manage'),
						$this->ctrl->getLinkTarget($this,'switchViewMode')
					);
					include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
					include_once './Services/Container/classes/class.ilContainerSortingSettings.php';
					include_once './Services/Container/classes/class.ilContainer.php';
					if((ilLinkResourceItems::lookupNumberOfLinks($this->object->getId()) > 1)
						and ilContainerSortingSettings::_lookupSortMode($this->object->getId()) == ilContainer::SORT_MANUAL)
					{
						$this->ctrl->setParameter($this,'switch_mode',self::VIEW_MODE_SORT);
						$ilTabs->addSubTab('id_content_ordering',
							$this->lng->txt('cntr_ordering'),
							$this->ctrl->getLinkTarget($this,'switchViewMode')
						);
					}
					
					$ilCtrl->clearParameters($this);
					$ilTabs->activateSubTab($a_active_subtab);
				}				
		}
		
		$ilTabs->activateTab('id_content');
	}
	
	
	/**
	* get tabs
	* @access	public
	*/
	function setTabs()
	{
		global $ilTabs, $lng, $ilHelp;
		
		$ilHelp->setScreenIdComponent("webr");
		
		if ($this->checkPermissionBool('read'))
		{
			$ilTabs->addTab("id_content",
				$lng->txt("content"),
				$this->ctrl->getLinkTarget($this, "view"));
		}
		
		if ($this->checkPermissionBool('visible'))
		{
			$ilTabs->addTab("id_info",
				$lng->txt("info_short"),
				$this->ctrl->getLinkTarget($this, "infoScreen"));
		}
		
		if($this->checkPermissionBool('write') and !$this->getCreationMode())
		{
			include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
			if(ilLinkResourceItems::lookupNumberOfLinks($this->object->getId()) > 1)
			{
				$ilTabs->addTab("id_settings",
					$lng->txt("settings"),
					$this->ctrl->getLinkTarget($this, "settings"));
			}
			
		}

		if ($this->checkPermissionBool('write'))
		{
			$ilTabs->addTab("id_history",
				$lng->txt("history"),
				$this->ctrl->getLinkTarget($this, "history"));
		}

		if ($this->checkPermissionBool('write'))
		{
			// Check if pear library is available
			if(@include_once('HTTP/Request.php'))
			{
				$ilTabs->addTab("id_link_check",
					$lng->txt("link_check"),
					$this->ctrl->getLinkTarget($this, "linkChecker"));
			}
		}

		if ($this->checkPermissionBool('write'))
		{
			$ilTabs->addTab("id_meta_data",
				$lng->txt("meta_data"),
				$this->ctrl->getLinkTargetByClass('ilmdeditorgui','listSection'));
		}

		if($this->checkPermissionBool('write'))
		{
			$ilTabs->addTab(
				'export',
				$this->lng->txt('export'),
				$this->ctrl->getLinkTargetByClass('ilexportgui','')
			);
		}

		// will add permission tab if needed
		parent::setTabs();
	}

	// PRIVATE
	function __prepareOutput()
	{
		// output objects
		$this->tpl->addBlockFile("CONTENT", "content", "tpl.adm_content.html");
		$this->tpl->addBlockFile("STATUSLINE", "statusline", "tpl.statusline.html");

		// output locator
		$this->__setLocator();

		// output message
		if ($this->message)
		{
			ilUtil::sendInfo($this->message);
		}

		// display infopanel if something happened
		ilUtil::infoPanel();;
	}

	function __setLocator()
	{
		global $tree;
		global $lng, $ilCtrl;

		$this->tpl->addBlockFile("LOCATOR", "locator", "tpl.locator.html", "Services/Locator");

		$counter = 0;
		
		//$this->tpl->touchBlock('locator_separator');
		//$this->tpl->touchBlock('locator_item');
		
		foreach ($tree->getPathFull($this->object->getRefId()) as $key => $row)
		{
			
			//if ($row["child"] == $tree->getRootId())
			//{
			//	continue;
			//}
			
			if($counter++)
			{
				$this->tpl->touchBlock('locator_separator_prefix');
			}

			if ($row["child"] > 0)
			{
				$this->tpl->setCurrentBlock("locator_img");
				$this->tpl->setVariable("IMG_SRC",
					ilUtil::getImagePath("icon_".$row["type"].".svg"));
				$this->tpl->setVariable("IMG_ALT",
					$lng->txt("obj_".$type));
				$this->tpl->parseCurrentBlock();
			}

			$this->tpl->setCurrentBlock("locator_item");

			if($row["type"] == 'webr')
			{
				$this->tpl->setVariable("ITEM",$this->object->getTitle());
				$this->tpl->setVariable("LINK_ITEM",$this->ctrl->getLinkTarget($this));
			}
			elseif ($row["child"] != $tree->getRootId())
			{
				$ilCtrl->setParameterByClass("ilrepositorygui", "ref_id", $row["child"]);
				$this->tpl->setVariable("ITEM", $row["title"]);
				$this->tpl->setVariable("LINK_ITEM",
					$ilCtrl->getLinkTargetByClass("ilrepositorygui", ""));
			}
			else
			{
				$ilCtrl->setParameterByClass("ilrepositorygui", "ref_id", $row["child"]);
				$this->tpl->setVariable("ITEM", $this->lng->txt("repository"));
				$this->tpl->setVariable("LINK_ITEM",
					$ilCtrl->getLinkTargetByClass("ilrepositorygui", ""));
			}
			$ilCtrl->setParameterByClass("ilrepositorygui", "ref_id", $_GET["ref_id"]);

			$this->tpl->parseCurrentBlock();
		}

		$this->tpl->setVariable("TXT_LOCATOR",$this->lng->txt("locator"));
		$this->tpl->parseCurrentBlock();
	}
	
	protected function handleSubItemLinks($a_target)
	{
		// #15647 - handle internal links
		include_once "Services/Form/classes/class.ilFormPropertyGUI.php";
		include_once "Services/Form/classes/class.ilLinkInputGUI.php";								
		if(ilLinkInputGUI::isInternalLink($a_target))			
		{						
			include_once("./Services/Link/classes/class.ilLink.php");
			
			// #10612
			$parts = explode("|", $a_target);
			
			if($parts[0] == "term")
			{
				// #16894
				return ilLink::_getStaticLink(
					0,
					"git",
					true,
					"&target=git_".$parts[1]
				);
			}
			
			if($parts[0] == "page")
			{
				$parts[0] = "pg";				
			}	
			
			$a_target = ilLink::_getStaticLink($parts[1], $parts[0]);
		}
		
		return $a_target;
	}
	
	function callDirectLink()
	{
		$obj_id = $this->object->getId();
						
		include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
		if(ilLinkResourceItems::_isSingular($obj_id))
		{
			$url = ilLinkResourceItems::_getFirstLink($obj_id);
			if($url["target"])
			{
				$url["target"] = $this->handleSubItemLinks($url["target"]);

				include_once './Modules/WebResource/classes/class.ilParameterAppender.php';
				if(ilParameterAppender::_isEnabled())
				{
				   $url = ilParameterAppender::_append($url);
				}		

				$this->redirectToLink($this->ref_id, $obj_id, $url["target"]);
			}
		}				
	}
	
	function callLink()
	{
		if($_REQUEST["link_id"])
		{		
			$obj_id = $this->object->getId();
			
			include_once './Modules/WebResource/classes/class.ilLinkResourceItems.php';
			$items = new ilLinkResourceItems($obj_id);
			$item = $items->getItem($_REQUEST["link_id"]);
			if($item["target"])
			{
				$item["target"] = $this->handleSubItemLinks($item["target"]);
				
				include_once './Modules/WebResource/classes/class.ilParameterAppender.php';
				if(ilParameterAppender::_isEnabled())
				{
				   $item = ilParameterAppender::_append($item);
				}

				$this->redirectToLink($this->ref_id, $obj_id, $item["target"]);
			}
		}
	}
	
	protected function redirectToLink($a_ref_id, $a_obj_id, $a_url)
	{
		global $ilUser;
		
		if($a_url)
		{
			require_once('Services/Tracking/classes/class.ilChangeEvent.php');
			ilChangeEvent::_recordReadEvent("webr", $a_ref_id, $a_obj_id,
				$ilUser->getId());
			
			ilUtil::redirect($a_url);
		}		
	}

	public static function _goto($a_target, $a_additional = null)
	{
		global $ilAccess, $ilErr, $lng;
				
		if($a_additional && substr($a_additional, -3) == "wsp")
		{
			$_GET["baseClass"] = "ilsharedresourceGUI";	
			$_GET["wsp_id"] = $a_target;		
			include("ilias.php");
			exit;
		}

		// Will be replaced in future releases by ilAccess::checkAccess()
		if ($ilAccess->checkAccess("read", "", $a_target))
		{
			ilUtil::redirect("ilias.php?baseClass=ilLinkResourceHandlerGUI&ref_id=$a_target");
		}
		else
		{
			// to do: force flat view
			if ($ilAccess->checkAccess("visible", "", $a_target))
			{
				ilUtil::redirect("ilias.php?baseClass=ilLinkResourceHandlerGUI&ref_id=".$a_target."&cmd=infoScreen");
			}
			else
			{
				if ($ilAccess->checkAccess("read", "", ROOT_FOLDER_ID))
				{
					ilUtil::sendFailure(sprintf($lng->txt("msg_no_perm_read_item"),
						ilObject::_lookupTitle(ilObject::_lookupObjId($a_target))), true);
					ilObjectGUI::_gotoRepositoryRoot();
				}
			}
		}

		$ilErr->raiseError($lng->txt("msg_no_perm_read"), $ilErr->FATAL);
	}
} // END class.ilObjLinkResource
?>
