<?php
/* Copyright (c) 1998-2013 ILIAS open source, Extended GPL, see docs/LICENSE */

require_once './Modules/Test/classes/inc.AssessmentConstants.php';
require_once './Modules/Test/classes/class.ilTestServiceGUI.php';
require_once 'Modules/TestQuestionPool/classes/class.ilAssQuestionHintTracking.php';

/**
 * Output class for assessment test evaluation
 *
 * The ilTestEvaluationGUI class creates the output for the ilObjTestGUI
 * class when authors evaluate a test. This saves some heap space because 
 * the ilObjTestGUI class will be much smaller then
 *
 * @author	Helmut Schottmüller <helmut.schottmueller@mac.com>
 * @author	Björn Heyser <bheyser@databay.de>
 * @author	Maximilian Becker <mbecker@databay.de>
 * 
 * @version		$Id$
 * 
 * @ingroup ModulesTest
 *
 * @ilCtrl_Calls ilTestEvaluationGUI: ilTestPassDetailsOverviewTableGUI
 * @ilCtrl_Calls ilTestEvaluationGUI: ilTestResultsToolbarGUI
 * @ilCtrl_Calls ilTestEvaluationGUI: ilTestPassDeletionConfirmationGUI
 */
class ilTestEvaluationGUI extends ilTestServiceGUI
{
	/**
	 * ilTestEvaluationGUI constructor
	 *
	 * The constructor takes possible arguments an creates an instance of the 
	 * ilTestEvaluationGUI object.
	 *
	 * @param ilObjTest $a_object Associated ilObjTest class
	 */
	public function __construct(ilObjTest $a_object)
	{
		parent::ilTestServiceGUI($a_object);
	}

	/**
	 * execute command
	 */
	public function &executeCommand()
	{
		$cmd = $this->ctrl->getCmd();
		$next_class = $this->ctrl->getNextClass($this);
		$this->ctrl->saveParameter($this, "sequence");
		$this->ctrl->saveParameter($this, "active_id");
		$cmd = $this->getCommand($cmd);
		switch($next_class)
		{
			case 'iltestpassdetailsoverviewtablegui':
				require_once 'Modules/Test/classes/tables/class.ilTestPassDetailsOverviewTableGUI.php';
				$tableGUI = new ilTestPassDetailsOverviewTableGUI($this->ctrl, $this, 'outUserPassDetails');
				$tableGUI->setIsPdfGenerationRequest($this->isPdfDeliveryRequest());
				$tableGUI->initFilter();
				$this->ctrl->forwardCommand($tableGUI);
				break;

			default:
				$ret =& $this->$cmd();
				break;
		}
		return $ret;
	}

	function &getHeaderNames()
	{
		$headernames = array();
		if ($this->object->getAnonymity())
		{
			array_push($headernames, $this->lng->txt("counter"));
		}
		else
		{
			array_push($headernames, $this->lng->txt("name"));
			array_push($headernames, $this->lng->txt("login"));
		}
		$additionalFields = $this->object->getEvaluationAdditionalFields();
		if (count($additionalFields))
		{
			foreach ($additionalFields as $fieldname)
			{
				array_push($headernames, $this->lng->txt($fieldname));
			}
		}
		array_push($headernames, $this->lng->txt("tst_reached_points"));
		array_push($headernames, $this->lng->txt("tst_mark"));
		if ($this->object->getECTSOutput())
		{
			array_push($headernames, $this->lng->txt("ects_grade"));
		}
		array_push($headernames, $this->lng->txt("tst_answered_questions"));
		array_push($headernames, $this->lng->txt("working_time"));
		array_push($headernames, $this->lng->txt("detailed_evaluation"));
		return $headernames;
	}
	
	function &getHeaderVars()
	{
		$headervars = array();
		if ($this->object->getAnonymity())
		{
			array_push($headervars, "counter");
		}
		else
		{
			array_push($headervars, "name");
			array_push($headervars, "login");
		}
		array_push($headervars, "resultspoints");
		array_push($headervars, "resultsmarks");
		if ($this->object->getECTSOutput())
		{
			array_push($headervars, "ects_grade");
		}
		array_push($headervars, "qworkedthrough");
		array_push($headervars, "timeofwork");
		array_push($headervars, "");
		return $headervars;
	}

	public function filterEvaluation()
	{
		include_once "./Modules/Test/classes/tables/class.ilEvaluationAllTableGUI.php";
		$table_gui = new ilEvaluationAllTableGUI($this, 'outEvaluation');
		$table_gui->writeFilterToSession();
		$this->ctrl->redirect($this, "outEvaluation");
	}

	public function resetfilterEvaluation()
	{
		include_once "./Modules/Test/classes/tables/class.ilEvaluationAllTableGUI.php";
		$table_gui = new ilEvaluationAllTableGUI($this, 'outEvaluation');
		$table_gui->resetFilter();
		$this->ctrl->redirect($this, "outEvaluation");
	}

	/**
	* Creates the evaluation output for the test
	*
	* @access public
	*/
	function outEvaluation()
	{
		/**
		 * @var $ilAcccess ilAccessHandler
		 * @var $ilToolbar ilToolbarGUI
		 */
		global $ilAccess, $ilToolbar;
		
		if ((!$ilAccess->checkAccess("tst_statistics", "", $this->ref_id)) && (!$ilAccess->checkAccess("write", "", $this->ref_id)))
		{
			// allow only evaluation access
			ilUtil::sendInfo($this->lng->txt("cannot_edit_test"), true);
			$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
		}

		include_once "./Modules/Test/classes/tables/class.ilEvaluationAllTableGUI.php";
		
		$table_gui = new ilEvaluationAllTableGUI(
				$this, 'outEvaluation', $this->object->getAnonymity(), $this->object->isOfferingQuestionHintsEnabled()
		);
		
		$data = array();
		$arrFilter = array();
		
		foreach ($table_gui->getFilterItems() as $item)
		{
			if ($item->getValue() !== false)
			{
				switch ($item->getPostVar())
				{
					case 'group':
					case 'name':
					case 'course':
						$arrFilter[$item->getPostVar()] = $item->getValue();
						break;
					case 'passed_only':
						$passedonly = $item->getChecked();
						break;
				}
			}
		}
		include_once "./Modules/Test/classes/class.ilTestEvaluationData.php";
		$eval = new ilTestEvaluationData($this->object);
		$eval->setFilterArray($arrFilter);
		$foundParticipants =& $eval->getParticipants();
		$counter = 1;
		if (count($foundParticipants) > 0)
		{
			if ($this->object->getECTSOutput())
			{
				$passed_array =& $this->object->getTotalPointsPassedArray();
			}
			foreach ($foundParticipants as $active_id => $userdata)
			{
				/* @var $userdata ilTestEvaluationUserData */
				
				$remove = FALSE;
				if ($passedonly)
				{
					$mark_obj = $this->object->getMarkSchema()->getMatchingMark($userdata->getReachedPointsInPercent());
					
					if( $mark_obj->getPassed() == FALSE || !$userdata->areObligationsAnswered() )
					{
						$remove = TRUE;
					}
				}
				if (!$remove)
				{
					// build the evaluation row
					$evaluationrow = array();
					if ($this->object->getAnonymity())
					{
						$evaluationrow['name'] = $counter;
						$evaluationrow['login'] = '';
					}
					else
					{
						$evaluationrow['name'] = $userdata->getName();
						if (strlen($userdata->getLogin()))
						{
							$evaluationrow['login'] = "[" . $userdata->getLogin() . "]";
						}
						else
						{
							$evaluationrow['login'] = '';
						}
					}

					$evaluationrow['reached'] = $userdata->getReached();
					$evaluationrow['max'] = $userdata->getMaxpoints();
					$evaluationrow['hint_count'] = $userdata->getRequestedHintsCountFromScoredPass();
					$percentage = $userdata->getReachedPointsInPercent();
					$mark = $this->object->getMarkSchema()->getMatchingMark($percentage);
					if (is_object($mark))
					{
						$evaluationrow['mark'] = $mark->getShortName();
					}
					if ($this->object->getECTSOutput())
					{
						$ects_mark = $this->object->getECTSGrade($passed_array, $userdata->getReached(), $userdata->getMaxPoints());
						$evaluationrow['ects_grade'] = $ects_mark;
					}
					$evaluationrow['answered'] = $userdata->getQuestionsWorkedThroughInPercent();
					$evaluationrow['questions_worked_through'] = $userdata->getQuestionsWorkedThrough();
					$evaluationrow['number_of_questions']      = $userdata->getNumberOfQuestions();
					$time_seconds = $userdata->getTimeOfWork();
					$time_hours    = floor($time_seconds/3600);
					$time_seconds -= $time_hours   * 3600;
					$time_minutes  = floor($time_seconds/60);
					$time_seconds -= $time_minutes * 60;
					$evaluationrow['working_time'] = sprintf("%02d:%02d:%02d", $time_hours, $time_minutes, $time_seconds);
					$this->ctrl->setParameter($this, "active_id", $active_id);
					$href = $this->ctrl->getLinkTarget($this, "detailedEvaluation");
					$detailed_evaluation = $this->lng->txt("detailed_evaluation_show");
					$evaluationrow['details'] = "<a class=\"il_ContainerItemCommand\" href=\"$href\">$detailed_evaluation</a>";
					$userfields = ilObjUser::_lookupFields($userdata->getUserID());
					$evaluationrow['gender'] = $userfields['gender'];
					$evaluationrow['email'] = $userfields['email'];
					$evaluationrow['institution'] = $userfields['institution'];
					$evaluationrow['street'] = $userfields['street'];
					$evaluationrow['city'] = $userfields['city'];
					$evaluationrow['zipcode'] = $userfields['zipcode'];
					$evaluationrow['country'] = $userfields['country'];
					$evaluationrow['sel_country'] = $userfields['sel_country'];
					$evaluationrow['department'] = $userfields['department'];
					$evaluationrow['matriculation'] = $userfields['matriculation'];
					$counter++;
					$data[] = $evaluationrow;
				}
			}
		}
		
		$table_gui->setData($data);
		if(count($foundParticipants) > 0)
		{
			$ilToolbar->setFormName('form_output_eval');
			$ilToolbar->setFormAction($this->ctrl->getFormAction($this, 'exportEvaluation'));
			require_once 'Services/Form/classes/class.ilSelectInputGUI.php';
			$export_type = new ilSelectInputGUI($this->lng->txt('exp_eval_data'), 'export_type');
			$options = array(
				'excel' => $this->lng->txt('exp_type_excel'),
				'csv'   => $this->lng->txt('exp_type_spss')
			);
			
			if(!$this->object->getAnonymity())
			{
				include_once 'Services/Certificate/classes/class.ilCertificate.php';
				include_once 'Modules/Test/classes/class.ilTestCertificateAdapter.php';
				if(ilCertificate::_isComplete(new ilTestCertificateAdapter($this->object)))
				{
					$options['certificate'] = $this->lng->txt('exp_type_certificate');
				}
			}

			$export_type->setOptions($options);
			
			$ilToolbar->addInputItem($export_type, true);
			require_once 'Services/UIComponent/Button/classes/class.ilSubmitButton.php';
			$button = ilSubmitButton::getInstance();
			$button->setCommand('exportEvaluation');
			$button->setCaption('export');
			$button->getOmitPreventDoubleSubmission();
			$ilToolbar->addButtonInstance($button);
		}

		$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print.css", "Modules/Test"), "print");
		if ($this->object->getShowSolutionAnswersOnly())
		{
			$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print_hide_content.css", "Modules/Test"), "print");
		}

		$this->tpl->setContent($table_gui->getHTML());
	}
	
	/**
	* Creates the detailed evaluation output for a selected participant
	*
	* Creates the detailed evaluation output for a selected participant
	*
	* @access public
	*/
	function detailedEvaluation()
	{
		global $ilAccess;
		
		if ((!$ilAccess->checkAccess("tst_statistics", "", $this->ref_id)) && (!$ilAccess->checkAccess("write", "", $this->ref_id)))
		{
			ilUtil::sendInfo($this->lng->txt("cannot_edit_test"), TRUE);
			$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
		}

		$this->tpl->addBlockFile("ADM_CONTENT", "adm_content", "tpl.il_as_tst_evaluation_details.html", "Modules/Test");

		$active_id = $_GET["active_id"];
		if (strlen($active_id) == 0)
		{
			ilUtil::sendInfo($this->lng->txt("detailed_evaluation_missing_active_id"), TRUE);
			$this->ctrl->redirect($this, "outEvaluation");
		}
		
		$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print.css", "Modules/Test"), "print");

		$data =& $this->object->getCompleteEvaluationData();
		$this->tpl->setVariable("TEXT_BACK", $this->lng->txt("back"));
		$this->tpl->setVariable("URL_BACK", $this->ctrl->getLinkTarget($this, "outEvaluation"));
		$this->tpl->setVariable("HEADING_DETAILED_EVALUATION", sprintf($this->lng->txt("detailed_evaluation_for"), 
			$data->getParticipant($active_id)->getName())
		);
		$this->tpl->setVariable("STATISTICAL_DATA", $this->lng->txt("statistical_data"));
		$this->tpl->setVariable("TXT_RESULTSPOINTS", $this->lng->txt("tst_stat_result_resultspoints"));
		$this->tpl->setVariable("VALUE_RESULTSPOINTS", $data->getParticipant($active_id)->getReached() . " " . strtolower($this->lng->txt("of")) . " " . $data->getParticipant($active_id)->getMaxpoints() . " (" . sprintf("%2.2f", $data->getParticipant($active_id)->getReachedPointsInPercent()) . " %" . ")");
		if (strlen($data->getParticipant($active_id)->getMark()))
		{
			$this->tpl->setVariable("TXT_RESULTSMARKS", $this->lng->txt("tst_stat_result_resultsmarks"));
			$this->tpl->setVariable("VALUE_RESULTSMARKS", $data->getParticipant($active_id)->getMark());
			if (strlen($data->getParticipant($active_id)->getECTSMark()))
			{
				$this->tpl->setVariable("TXT_ECTS", $this->lng->txt("ects_grade"));
				$this->tpl->setVariable("VALUE_ECTS", $data->getParticipant($active_id)->getECTSMark());
			}
		}
		
		if( $this->object->isOfferingQuestionHintsEnabled() )
		{
			$this->tpl->setVariable("TXT_REQUESTED_HINTS_COUNT", $this->lng->txt("tst_question_hints_requested_hint_count_header"));
			$this->tpl->setVariable("VALUE_REQUESTED_HINTS_COUNT", $data->getParticipant($active_id)->getRequestedHintsCountFromScoredPass());
		}
		
		$this->tpl->setVariable("TXT_QWORKEDTHROUGH", $this->lng->txt("tst_stat_result_qworkedthrough"));
		$this->tpl->setVariable("VALUE_QWORKEDTHROUGH", $data->getParticipant($active_id)->getQuestionsWorkedThrough() . " " . strtolower($this->lng->txt("of")) . " " . $data->getParticipant($active_id)->getNumberOfQuestions() . " (" . sprintf("%2.2f", $data->getParticipant($active_id)->getQuestionsWorkedThroughInPercent()) . " %" . ")");

		$this->tpl->setVariable("TXT_TIMEOFWORK", $this->lng->txt("tst_stat_result_timeofwork"));
		$time_seconds = $data->getParticipant($active_id)->getTimeOfWork();
		$atime_seconds = $data->getParticipant($active_id)->getNumberOfQuestions() ? $time_seconds / $data->getParticipant($active_id)->getNumberOfQuestions() : 0;
		$time_hours    = floor($time_seconds/3600);
		$time_seconds -= $time_hours   * 3600;
		$time_minutes  = floor($time_seconds/60);
		$time_seconds -= $time_minutes * 60;
		$this->tpl->setVariable("VALUE_TIMEOFWORK", sprintf("%02d:%02d:%02d", $time_hours, $time_minutes, $time_seconds));
		$this->tpl->setVariable("TXT_ATIMEOFWORK", $this->lng->txt("tst_stat_result_atimeofwork"));
		$time_hours    = floor($atime_seconds/3600);
		$atime_seconds -= $time_hours   * 3600;
		$time_minutes  = floor($atime_seconds/60);
		$atime_seconds -= $time_minutes * 60;
		$this->tpl->setVariable("VALUE_ATIMEOFWORK", sprintf("%02d:%02d:%02d", $time_hours, $time_minutes, $atime_seconds));
		$this->tpl->setVariable("TXT_FIRSTVISIT", $this->lng->txt("tst_stat_result_firstvisit"));
		#$this->tpl->setVariable("VALUE_FIRSTVISIT", 
		#	date($this->lng->text["lang_dateformat"] . " " . $this->lng->text["lang_timeformat"], $data->getParticipant($active_id)->getFirstVisit())
		#);
		$this->tpl->setVariable('VAL_FIRST_VISIT',ilDatePresentation::formatDate(
			new ilDateTime($data->getParticipant($active_id)->getFirstVisit(),IL_CAL_UNIX)));
		$this->tpl->setVariable("TXT_LASTVISIT", $this->lng->txt("tst_stat_result_lastvisit"));
		#$this->tpl->setVariable("VALUE_LASTVISIT",
		#	date($this->lng->text["lang_dateformat"] . " " . $this->lng->text["lang_timeformat"], $data->getParticipant($active_id)->getLastVisit())
		#);
		$this->tpl->setVariable('VAL_FIRST_VISIT',ilDatePresentation::formatDate(
			new ilDateTime($data->getParticipant($active_id)->getLastVisit(),IL_CAL_UNIX)));
		
		$this->tpl->setVariable("TXT_NROFPASSES", $this->lng->txt("tst_nr_of_passes"));
		$this->tpl->setVariable("VALUE_NROFPASSES", $data->getParticipant($active_id)->getLastPass() + 1);
		$this->tpl->setVariable("TXT_SCOREDPASS", $this->lng->txt("scored_pass"));
		if ($this->object->getPassScoring() == SCORE_BEST_PASS)
		{
			$this->tpl->setVariable("VALUE_SCOREDPASS", $data->getParticipant($active_id)->getBestPass() + 1);
		}
		else
		{
			$this->tpl->setVariable("VALUE_SCOREDPASS", $data->getParticipant($active_id)->getLastPass() + 1);
		}
		
		$median = $data->getStatistics()->getStatistics()->median();
		$pct = $data->getParticipant($active_id)->getMaxpoints() ? ($median / $data->getParticipant($active_id)->getMaxpoints()) * 100.0 : 0;
		$mark = $this->object->mark_schema->getMatchingMark($pct);
		if (is_object($mark))
		{
			$this->tpl->setVariable("TXT_MARK_MEDIAN", $this->lng->txt("tst_stat_result_mark_median"));
			$this->tpl->setVariable("VALUE_MARK_MEDIAN", $mark->getShortName());
		}

		$this->tpl->setVariable("TXT_RANK_PARTICIPANT", $this->lng->txt("tst_stat_result_rank_participant"));
		$this->tpl->setVariable("VALUE_RANK_PARTICIPANT", $data->getStatistics()->getStatistics()->rank($data->getParticipant($active_id)->getReached()));
		$this->tpl->setVariable("TXT_RANK_MEDIAN", $this->lng->txt("tst_stat_result_rank_median"));
		$this->tpl->setVariable("VALUE_RANK_MEDIAN", $data->getStatistics()->getStatistics()->rank_median());
		$this->tpl->setVariable("TXT_TOTAL_PARTICIPANTS", $this->lng->txt("tst_stat_result_total_participants"));
		$this->tpl->setVariable("VALUE_TOTAL_PARTICIPANTS", $data->getStatistics()->getStatistics()->count());
		$this->tpl->setVariable("TXT_RESULT_MEDIAN", $this->lng->txt("tst_stat_result_median"));
		$this->tpl->setVariable("VALUE_RESULT_MEDIAN", $median);

		for ($pass = 0; $pass <= $data->getParticipant($active_id)->getLastPass(); $pass++)
		{
			$finishdate = $this->object->getPassFinishDate($active_id, $pass);
			if ($finishdate > 0)
			{
				$this->tpl->setCurrentBlock("question_header");
				$this->tpl->setVariable("TXT_QUESTION_DATA", sprintf($this->lng->txt("tst_eval_question_points"), $pass+1));
				$this->tpl->parseCurrentBlock();
				global $ilAccess;
				if (($ilAccess->checkAccess("write", "", $_GET["ref_id"])))
				{
					$this->tpl->setCurrentBlock("question_footer");
					$this->tpl->setVariable("TEXT_TO_DETAILED_RESULTS", $this->lng->txt("tst_show_answer_sheet"));
					$this->ctrl->setParameter($this, "statistics", "1");
					$this->ctrl->setParameter($this, "active_id", $active_id);
					$this->ctrl->setParameter($this, "pass", $pass);
					$this->tpl->setVariable("URL_TO_DETAILED_RESULTS", $this->ctrl->getLinkTarget($this, "outParticipantsPassDetails"));
					$this->tpl->parseCurrentBlock();
				}
				$questions = $data->getParticipant($active_id)->getQuestions($pass);
				if (!is_array($questions))
				{
					$questions = $data->getParticipant($active_id)->getQuestions(0);
				}
				$counter = 1;
				foreach ($questions as $question)
				{
					$this->tpl->setCurrentBlock("question_row");
					$this->tpl->setVariable("QUESTION_COUNTER", $counter);
					$this->tpl->setVariable("QUESTION_ID", $question["id"]);
					$this->tpl->setVariable("QUESTION_ID_TXT", $this->lng->txt('question_id_short'));
					$this->tpl->setVariable("QUESTION_TITLE", $data->getQuestionTitle($question["id"]));
					$answeredquestion = $data->getParticipant($active_id)->getPass($pass)->getAnsweredQuestionByQuestionId($question["id"]);
					if (is_array($answeredquestion))
					{
						$percent = $answeredquestion["points"] ? $answeredquestion["reached"] / $answeredquestion["points"] * 100.0 : 0;
						$this->tpl->setVariable("QUESTION_POINTS", $answeredquestion["reached"] . " " . strtolower($this->lng->txt("of")) . " " . $answeredquestion["points"] . " (" . sprintf("%.2f", $percent) . " %)");
					}
					else
					{
						$this->tpl->setVariable("QUESTION_POINTS",  "0 " . strtolower($this->lng->txt("of")) . " " . $question["points"] . " (" . sprintf("%.2f", 0) . " %) - " . $this->lng->txt("question_not_answered"));
					}
					$this->tpl->parseCurrentBlock();
					$counter++;
				}
				$this->tpl->touchBlock("question_stats");
			}
		}
	}
	
	/**
	 * Creates a PDF representation of the answers for a given question in a test
	 *
	 */
	public function exportQuestionForAllParticipants()
	{
		$this->getQuestionResultForTestUsers($_GET["qid"], $this->object->getTestId());
	}
	
	/**
	 * Creates a ZIP file containing all file uploads for a given question in a test
	 *
	 */
	public function exportFileUploadsForAllParticipants()
	{
		require_once './Modules/TestQuestionPool/classes/class.assQuestion.php';
		$question_object = assQuestion::_instanciateQuestion($_GET["qid"]);
		if ( $question_object instanceof ilObjFileHandlingQuestionType )
		{
			$question_object->getFileUploadZIPFile($this->object->getTestId());
		}
		else
		{
			$this->ctrl->redirect($this, "singleResults");
		}
	}
	
/**
* Output of anonymous aggregated results for the test
*
* Output of anonymous aggregated results for the test
*
* @access public
*/
	function eval_a()
	{
		/**
		 * @var $ilAccess ilAccessHandler
		 * @var $ilToolbar ilToolbarGUI
		 */
		global $ilAccess, $ilToolbar;

		if ((!$ilAccess->checkAccess("tst_statistics", "", $this->ref_id)) && (!$ilAccess->checkAccess("write", "", $this->ref_id)))
		{
			// allow only evaluation access
			ilUtil::sendInfo($this->lng->txt("cannot_edit_test"), true);
			$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
		}

		$this->tpl->addBlockFile("ADM_CONTENT", "adm_content", "tpl.il_as_tst_eval_anonymous_aggregation.html", "Modules/Test");
		$eval =& $this->object->getCompleteEvaluationData();
		$data = array();
		$foundParticipants =& $eval->getParticipants();
		if (count($foundParticipants)) 
		{
			$ilToolbar->setFormName('form_output_eval');
			$ilToolbar->setFormAction($this->ctrl->getFormAction($this, 'exportAggregatedResults'));
			require_once 'Services/Form/classes/class.ilSelectInputGUI.php';
			$export_type = new ilSelectInputGUI($this->lng->txt('exp_eval_data'), 'export_type');
			$export_type->setOptions(array(
				'excel' => $this->lng->txt('exp_type_excel'),
				'csv'   => $this->lng->txt('exp_type_spss')
			));
			$ilToolbar->addInputItem($export_type, true);
			require_once 'Services/UIComponent/Button/classes/class.ilSubmitButton.php';
			$button = ilSubmitButton::getInstance();
			$button->setCommand('exportAggregatedResults');
			$button->setCaption('export');
			$button->getOmitPreventDoubleSubmission();
			$ilToolbar->addButtonInstance($button);

			array_push($data, array(
				'result' => $this->lng->txt("tst_eval_total_persons"),
				'value'  => count($foundParticipants)
			));
			$total_finished = $this->object->evalTotalFinished();
			array_push($data, array(
				'result' => $this->lng->txt("tst_eval_total_finished"),
				'value'  => $total_finished
			));
			$average_time = $this->object->evalTotalStartedAverageTime();
			$diff_seconds = $average_time;
			$diff_hours    = floor($diff_seconds/3600);
			$diff_seconds -= $diff_hours   * 3600;
			$diff_minutes  = floor($diff_seconds/60);
			$diff_seconds -= $diff_minutes * 60;
			array_push($data, array(
				'result' => $this->lng->txt("tst_eval_total_finished_average_time"),
				'value'  => sprintf("%02d:%02d:%02d", $diff_hours, $diff_minutes, $diff_seconds)
			));
			$total_passed = 0;
			$total_passed_reached = 0;
			$total_passed_max = 0;
			$total_passed_time = 0;
			foreach ($foundParticipants as $userdata)
			{
				if ($userdata->getPassed()) 
				{
					$total_passed++;
					$total_passed_reached += $userdata->getReached();
					$total_passed_max += $userdata->getMaxpoints();
					$total_passed_time += $userdata->getTimeOfWork();
				}
			}
			$average_passed_reached = $total_passed ? $total_passed_reached / $total_passed : 0;
			$average_passed_max = $total_passed ? $total_passed_max / $total_passed : 0;
			$average_passed_time = $total_passed ? $total_passed_time / $total_passed : 0;
			array_push($data, array(
				'result' => $this->lng->txt("tst_eval_total_passed"),
				'value'  => $total_passed
			));
			array_push($data, array(
				'result' => $this->lng->txt("tst_eval_total_passed_average_points"),
				'value'  => sprintf("%2.2f", $average_passed_reached) . " " . strtolower($this->lng->txt("of")) . " " . sprintf("%2.2f", $average_passed_max)
			));
			$average_time = $average_passed_time;
			$diff_seconds = $average_time;
			$diff_hours    = floor($diff_seconds/3600);
			$diff_seconds -= $diff_hours   * 3600;
			$diff_minutes  = floor($diff_seconds/60);
			$diff_seconds -= $diff_minutes * 60;
			array_push($data, array(
				'result' => $this->lng->txt("tst_eval_total_passed_average_time"),
				'value'  => sprintf("%02d:%02d:%02d", $diff_hours, $diff_minutes, $diff_seconds)
			));
		} 

		include_once "./Modules/Test/classes/tables/class.ilTestAggregatedResultsTableGUI.php";
		$table_gui = new ilTestAggregatedResultsTableGUI($this, 'eval_a');
		$table_gui->setData($data);
		$this->tpl->setVariable('AGGREGATED_RESULTS', $table_gui->getHTML());	
		
		$rows = array();
		foreach ($eval->getQuestionTitles() as $question_id => $question_title)
		{
			$answered = 0;
			$reached = 0;
			$max = 0;
			foreach ($foundParticipants as $userdata)
			{
				for ($i = 0; $i <= $userdata->getLastPass(); $i++)
				{
					if (is_object($userdata->getPass($i)))
					{
						$question =& $userdata->getPass($i)->getAnsweredQuestionByQuestionId($question_id);
						if (is_array($question))
						{
							$answered++;
							$reached += $question["reached"];
							$max += $question["points"];
						}
					}
				}
			}
			$percent = $max ? $reached/$max * 100.0 : 0;
			$counter++;
			$this->ctrl->setParameter($this, "qid", $question_id);

			$points_reached = ($answered ? $reached / $answered : 0);
			$points_max     = ($answered ? $max / $answered : 0);
			array_push($rows,
				array(
					'qid'            => $question_id,
					'title'          => $question_title,
					'points'         => $points_reached,
					'points_reached' => $points_reached,
					'points_max'     => $points_max,
					'percentage'     => (float)$percent,
					'answers'        => $answered
				)
			);
		}
		include_once "./Modules/Test/classes/tables/class.ilTestAverageReachedPointsTableGUI.php";
		$table_gui = new ilTestAverageReachedPointsTableGUI($this, 'eval_a');
		$table_gui->setData($rows);
		$this->tpl->setVariable('TBL_AVG_REACHED', $table_gui->getHTML());	
	}

	/**
	 * Exports the evaluation data to a selected file format
	 */
	public function exportEvaluation()
	{
		$filterby = "";
		if (array_key_exists("g_filterby", $_GET))
		{
			$filterby = $_GET["g_filterby"];
		}

		$filtertext = "";
		if (array_key_exists("g_userfilter", $_GET))
		{
			$filtertext = $_GET["g_userfilter"];
		}

		$passedonly = FALSE;
		if (array_key_exists("g_passedonly", $_GET))
		{
			if ($_GET["g_passedonly"] == 1)
			{
				$passedonly = TRUE;
			}
		}

		switch ($_POST["export_type"])
		{
			case "excel":
				require_once './Modules/Test/classes/class.ilTestExport.php';
				$exportObj = new ilTestExport($this->object, "results");
				$exportObj->exportToExcel($deliver = TRUE, $filterby, $filtertext, $passedonly);
				break;

			case "csv":
				require_once './Modules/Test/classes/class.ilTestExport.php';
				$exportObj = new ilTestExport($this->object, "results");
				$exportObj->exportToCSV($deliver = TRUE, $filterby, $filtertext, $passedonly);
				break;

			case "certificate":
				if ($passedonly)
				{
					$this->ctrl->setParameterByClass("iltestcertificategui", "g_passedonly", "1");
				}
				if (strlen($filtertext))
				{
					$this->ctrl->setParameterByClass("iltestcertificategui", "g_userfilter", $filtertext);
				}
				$this->ctrl->redirect($this, "exportCertificate");
				break;
		}
	}

	/**
	* Exports the aggregated results
	*
	* @access public
	*/
	function exportAggregatedResults()
	{
		switch ($_POST["export_type"])
		{
			case "excel":
				include_once "./Modules/Test/classes/class.ilTestExport.php";
				$exportObj = new ilTestExport($this->object, "aggregated");
				$exportObj->exportToExcel($deliver = TRUE);
				break;
			case "csv":
				include_once "./Modules/Test/classes/class.ilTestExport.php";
				$exportObj = new ilTestExport($this->object, "aggregated");
				$exportObj->exportToCSV($deliver = TRUE);
				break;
		}
	}

	/**
	* Exports the user results as PDF certificates using
	* XSL-FO via XML:RPC calls
	*
	* @access public
	*/
	public function exportCertificate()
	{
		global $ilUser;
		
		include_once "./Services/Utilities/classes/class.ilUtil.php";
		include_once "./Services/Certificate/classes/class.ilCertificate.php";
		include_once "./Modules/Test/classes/class.ilTestCertificateAdapter.php";
		$certificate = new ilCertificate(new ilTestCertificateAdapter($this->object));
		$archive_dir = $certificate->createArchiveDirectory();
		$total_users = array();
		$total_users =& $this->object->evalTotalPersonsArray();
		if (count($total_users))
		{
			foreach ($total_users as $active_id => $name)
			{
				$user_id = $this->object->_getUserIdFromActiveId($active_id);
				$pdf = $certificate->outCertificate(
					array(
						"active_id" => $active_id,
						"userfilter" => $userfilter,
						"passedonly" => $passedonly
					),
					FALSE
				);
				if (strlen($pdf))
				{
					$certificate->addPDFtoArchiveDirectory($pdf, $archive_dir, $user_id . "_" . str_replace(" ", "_", ilUtil::getASCIIFilename($name)) . ".pdf");
				}
			}
			$zipArchive = $certificate->zipCertificatesInArchiveDirectory($archive_dir, TRUE);
		}

	}
	
	/**
	 * Returns the ID of a question for evaluation purposes. If a question id and the id of the
	 * original question are given, this function returns the original id, otherwise the  question id
	 *
	 * @return int question or original id
	 **/
	function getEvaluationQuestionId($question_id, $original_id = "")
	{
		if ($original_id > 0)
		{
			return $original_id;
		}
		else
		{
			return $question_id;
		}
	}
	
	/**
	* Output of the pass details of an existing test pass for the test statistics
	*
	* Output of the pass details of an existing test pass for the test statistics
	*
	* @access public
	*/
	function outParticipantsPassDetails()
	{
		global $ilTabs, $ilAccess;

		if (!$ilAccess->checkAccess('write', '', $this->ref_id))
		{
			// allow only write access
			ilUtil::sendInfo($this->lng->txt('no_permission'), true);
			$this->ctrl->redirectByClass('ilObjTestGUI', 'infoScreen');
		}

		$this->ctrl->saveParameter($this, "active_id");
		$active_id = (int)$_GET["active_id"];
		$testSession = $this->testSessionFactory->getSession($active_id);

		// protect actives from other tests
		if( $testSession->getTestId() != $this->object->getTestId() )
		{
			ilUtil::sendInfo($this->lng->txt('no_permission'), true);
			$this->ctrl->redirectByClass('ilObjTestGUI', 'infoScreen');
		}
		
		$this->ctrl->saveParameter($this, "pass");
		$pass = (int)$_GET["pass"];

		if ( isset($_GET['statistics']) && $_GET['statistics'] == 1)
		{
			$this->ctrl->setParameterByClass("ilTestEvaluationGUI", "active_id", $active_id);

			$ilTabs->setBackTarget(
				$this->lng->txt('back'), $this->ctrl->getLinkTargetByClass('ilTestEvaluationGUI', 'detailedEvaluation')
			);
		}
		elseif ($this->object->getNrOfTries() == 1)
		{
			$ilTabs->setBackTarget(
				$this->lng->txt('back'), $this->ctrl->getLinkTargetByClass('ilobjtestgui', 'participants')
			);
		}
		else
		{
			$ilTabs->setBackTarget(
				$this->lng->txt('tst_results_back_overview'), $this->ctrl->getLinkTarget($this, 'outParticipantsResultsOverview')
			);
		}
		
		$result_array =& $this->object->getTestResult($active_id, $pass);
		
		$overview = $this->getPassDetailsOverview($result_array, $active_id, $pass, $this, "outParticipantsPassDetails", '', true);
		$user_data = $this->getResultsUserdata($testSession, $active_id, FALSE);
		$user_id = $this->object->_getUserIdFromActiveId($active_id);

		$template = new ilTemplate("tpl.il_as_tst_pass_details_overview_participants.html", TRUE, TRUE, "Modules/Test");

		require_once 'Modules/Test/classes/toolbars/class.ilTestResultsToolbarGUI.php';
		$toolbar = new ilTestResultsToolbarGUI($this->ctrl, $this->tpl, $this->lng);

		$this->ctrl->setParameter($this, 'pdf', '1');
		$toolbar->setPdfExportLinkTarget( $this->ctrl->getLinkTarget($this, 'outParticipantsPassDetails') );
		$this->ctrl->setParameter($this, 'pdf', '');

		if( isset($_GET['show_best_solutions']) )
		{
			$_SESSION['tst_results_show_best_solutions'] = true;
		}
		elseif( isset($_GET['hide_best_solutions']) )
		{
			$_SESSION['tst_results_show_best_solutions'] = false;
		}
		elseif( !isset($_SESSION['tst_results_show_best_solutions']) )
		{
			$_SESSION['tst_results_show_best_solutions'] = false;
		}

		if( $_SESSION['tst_results_show_best_solutions'] )
		{
			$this->ctrl->setParameter($this, 'hide_best_solutions', '1');
			$toolbar->setHideBestSolutionsLinkTarget($this->ctrl->getLinkTarget($this, 'outParticipantsPassDetails'));
			$this->ctrl->setParameter($this, 'hide_best_solutions', '');
		}
		else
		{
			$this->ctrl->setParameter($this, 'show_best_solutions', '1');
			$toolbar->setShowBestSolutionsLinkTarget($this->ctrl->getLinkTarget($this, 'outParticipantsPassDetails'));
			$this->ctrl->setParameter($this, 'show_best_solutions', '');
		}

		$toolbar->build();
		$template->setVariable('RESULTS_TOOLBAR', $this->ctrl->getHTML($toolbar));

		if( $this->isGradingMessageRequired() && $this->object->getNrOfTries() == 1 )
		{
			$template->setCurrentBlock('grading_message');
			$template->setVariable('GRADING_MESSAGE', $this->getGradingMessage($active_id));
			$template->parseCurrentBlock();
		}

		$list_of_answers = $this->getPassListOfAnswers($result_array, $active_id, $pass, $_SESSION['tst_results_show_best_solutions'], false, false, false, true);
		$template->setVariable("LIST_OF_ANSWERS", $list_of_answers);
		$template->setVariable("TEXT_RESULTS", $this->lng->txt("tst_results"));
		$template->setVariable("FORMACTION", $this->ctrl->getFormAction($this));
		$template->setVariable("PASS_DETAILS", $overview);
		$template->setVariable("USER_DATA", $user_data);
		$uname = $this->object->userLookupFullName($user_id);
		$template->setVariable("TEXT_HEADING", sprintf($this->lng->txt("tst_result_user_name_pass"), $pass + 1, $uname));

		$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print.css", "Modules/Test"), "print");
		if ($this->object->getShowSolutionAnswersOnly())
		{
			$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print_hide_content.css", "Modules/Test"), "print");
		}

		if( $this->isPdfDeliveryRequest() )
		{
			//$this->object->deliverPDFfromHTML($template->get());
			require_once 'class.ilTestPDFGenerator.php';
			ilTestPDFGenerator::generatePDF($template->get(), ilTestPDFGenerator::PDF_OUTPUT_DOWNLOAD, $this->object->getTitle());
		}
		else
		{
			$this->tpl->setVariable("ADM_CONTENT", $template->get());
		}
	}

	/**
	* Output of the pass overview for a test called from the statistics
	*
	* @access public
	*/
	function outParticipantsResultsOverview()
	{
		global $ilTabs, $ilAccess;
		
		if (!$ilAccess->checkAccess('write', '', $this->ref_id))
		{
			// allow only write access
			ilUtil::sendInfo($this->lng->txt('no_permission'), true);
			$this->ctrl->redirectByClass('ilObjTestGUI', 'infoScreen');
		}

		$active_id = (int)$_GET["active_id"];
		$testSession = $this->testSessionFactory->getSession($active_id);

		// protect actives from other tests
		if( $testSession->getTestId() != $this->object->getTestId() )
		{
			ilUtil::sendInfo($this->lng->txt('no_permission'), true);
			$this->ctrl->redirectByClass('ilObjTestGUI', 'infoScreen');
		}

		if ($this->object->getNrOfTries() == 1)
		{
			$this->ctrl->setParameter($this, "active_id", $active_id);
			$this->ctrl->setParameter($this, "pass", ilObjTest::_getResultPass($active_id));
			$this->ctrl->redirect($this, "outParticipantsPassDetails");
		}

		$ilTabs->setBackTarget(
			$this->lng->txt('back'), $this->ctrl->getLinkTargetByClass('ilobjtestgui', 'participants')
		);

		$template = new ilTemplate("tpl.il_as_tst_pass_overview_participants.html", TRUE, TRUE, "Modules/Test");

		require_once 'Modules/Test/classes/toolbars/class.ilTestResultsToolbarGUI.php';
		$toolbar = new ilTestResultsToolbarGUI($this->ctrl, $this->tpl, $this->lng);

		$this->ctrl->setParameter($this, 'pdf', '1');
		$toolbar->setPdfExportLinkTarget( $this->ctrl->getLinkTarget($this, __FUNCTION__) );
		$this->ctrl->setParameter($this, 'pdf', '');

		$toolbar->build();
		$template->setVariable('RESULTS_TOOLBAR', $this->ctrl->getHTML($toolbar));

		$overview = $this->getPassOverview($active_id, "iltestevaluationgui", "outParticipantsPassDetails");
		$template->setVariable("PASS_OVERVIEW", $overview);

		$user_id = $this->object->_getUserIdFromActiveId($active_id);
		$user_data = $this->getResultsUserdata($testSession, $active_id);
		$template->setVariable("USER_DATA", $user_data);
		$template->setVariable("TEXT_OVERVIEW", $this->lng->txt("tst_results_overview"));

		if( $this->isGradingMessageRequired() )
		{
			$template->setCurrentBlock('grading_message');
			$template->setVariable('GRADING_MESSAGE', $this->getGradingMessage($active_id));
			$template->parseCurrentBlock();
		}

		$template->setVariable("TEXT_RESULTS", $this->lng->txt("tst_results"));
		$template->parseCurrentBlock();

		$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print.css", "Modules/Test"), "print");
		if ($this->object->getShowSolutionAnswersOnly())
		{
			$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print_hide_content.css", "Modules/Test"), "print");
		}

		if (array_key_exists("pdf", $_GET) && ($_GET["pdf"] == 1))
		{
			//$this->object->deliverPDFfromHTML($template->get(), $this->object->getTitle());

			$name = ilObjUser::_lookupName($user_id);
			$filename = $name['lastname'] . '_' . $name['firstname'] . '_' . $name['login'] . '__'. $this->object->getTitle();
			require_once 'class.ilTestPDFGenerator.php';
			ilTestPDFGenerator::generatePDF($template->get(), ilTestPDFGenerator::PDF_OUTPUT_DOWNLOAD, $filename);
			//ilUtil::deliverData($file, ilUtil::getASCIIFilename($this->object->getTitle()) . ".pdf", "application/pdf", false, true);
			//$template->setVariable("PDF_FILE_LOCATION", $filename);
		}
		else
		{
			$this->tpl->setVariable("ADM_CONTENT", $template->get());
		}
	}

	public function outUserPassDetailsSetTableFilter()
	{
		$tableGUI = $this->buildPassDetailsOverviewTableGUI($this, 'outUserPassDetails');
		$tableGUI->initFilter();
		$tableGUI->resetOffset();
		$tableGUI->writeFilterToSession();
		$this->outUserPassDetails();
	}

	public function outUserPassDetailsResetTableFilter()
	{
		$tableGUI = $this->buildPassDetailsOverviewTableGUI($this, 'outUserPassDetails');
		$tableGUI->initFilter();
		$tableGUI->resetOffset();
		$tableGUI->resetFilter();
		$this->outUserPassDetails();
	}

	/**
	 * Output of the pass details of an existing test pass for the active test participant
	 *
	 * @access public
	 */
	function outUserPassDetails()
	{
		global $ilTabs, $ilUser;

		if ($this->object->getNrOfTries() == 1)
		{
			$ilTabs->setBackTarget($this->lng->txt('tst_results_back_introduction'), $this->ctrl->getLinkTargetByClass('ilObjTestGUI', 'infoScreen'));
		}
		else
		{
			$ilTabs->setBackTarget($this->lng->txt('tst_results_back_overview'), $this->ctrl->getLinkTarget($this, 'outUserResultsOverview'));
		}

		$testSession = $this->testSessionFactory->getSession();

		if( !$this->object->getShowPassDetails() )
		{
			#$executable = $this->object->isExecutable($testSession, $ilUser->getId());

			#if($executable["executable"])
			#{
				$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
			#}
		}

		$active_id = $testSession->getActiveId();
		$user_id = $testSession->getUserId();

		$this->ctrl->saveParameter($this, "pass");
		$pass = $_GET["pass"];

		$result_array = $this->getFilteredTestResult($active_id, $pass);

		$command_solution_details = "";
		if ($this->object->getShowSolutionDetails())
		{
			$command_solution_details = "outCorrectSolution";
		}
		$questionAnchorNav = $this->object->canShowSolutionPrintview();

		$tpl = new ilTemplate('tpl.il_as_tst_pass_details_overview_participants.html', true, true, "Modules/Test");

		if( !$this->isPdfDeliveryRequest() )
		{
			require_once 'Modules/Test/classes/toolbars/class.ilTestResultsToolbarGUI.php';
			$toolbar = new ilTestResultsToolbarGUI($this->ctrl, $this->tpl, $this->lng);

			$this->ctrl->setParameter($this, 'pdf', '1');
			$toolbar->setPdfExportLinkTarget( $this->ctrl->getLinkTarget($this, 'outUserPassDetails') );
			$this->ctrl->setParameter($this, 'pdf', '');

			include_once './Services/WebServices/RPC/classes/class.ilRPCServerSettings.php';
			if( $this->object->canShowCertificate($testSession, $user_id, $active_id) )
			{
				$toolbar->setCertificateLinkTarget($this->ctrl->getLinkTarget($this, 'outCertificate'));
			}

			$toolbar->build();

			$tpl->setVariable('RESULTS_TOOLBAR', $this->ctrl->getHTML($toolbar));

			$tpl->setCurrentBlock('signature');
			$tpl->setVariable("SIGNATURE", $this->getResultsSignature());
			$tpl->parseCurrentBlock();
			
			if ($this->object->isShowExamIdInTestResultsEnabled())
			{
				$tpl->setCurrentBlock('exam_id');
				$tpl->setVariable('EXAM_ID', $this->object->lookupExamId(
					$testSession->getActiveId(), $pass
				));
				$tpl->setVariable('EXAM_ID_TXT', $this->lng->txt('exam_id'));
				$tpl->parseCurrentBlock();
			}
		}

		if( $this->isGradingMessageRequired() && $this->object->getNrOfTries() == 1 )
		{
			$tpl->setCurrentBlock('grading_message');
			$tpl->setVariable('GRADING_MESSAGE', $this->getGradingMessage($active_id));
			$tpl->parseCurrentBlock();
		}

		$overview = $this->getPassDetailsOverview(
			$result_array, $active_id, $pass, $this, "outUserPassDetails", $command_solution_details, $questionAnchorNav
		);
		$tpl->setVariable("PASS_DETAILS", $overview);

		if( $this->object->canShowSolutionPrintview() )
		{
			$list_of_answers = $this->getPassListOfAnswers(
				$result_array, $active_id, $pass, $this->object->getShowSolutionListComparison(),
				false, false, false, true
			);
			$tpl->setVariable("LIST_OF_ANSWERS", $list_of_answers);
		}

		$tpl->setVariable("TEXT_RESULTS", $this->lng->txt("tst_results"));
		$tpl->setVariable("FORMACTION", $this->ctrl->getFormAction($this));

		$uname = $this->object->userLookupFullName($user_id, TRUE);
		$user_data = $this->getResultsUserdata($testSession, $active_id, TRUE);
		if ($this->object->getAnonymity())
		{
			$tpl->setVariable("TEXT_HEADING", $this->lng->txt("tst_result_pass"));
		}
		else
		{
			$tpl->setVariable("TEXT_HEADING", sprintf($this->lng->txt("tst_result_user_name_pass"), $pass + 1, $uname));
			$tpl->setVariable("USER_DATA", $user_data);
		}

		$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print.css", "Modules/Test"), "print");
		if ($this->object->getShowSolutionAnswersOnly())
		{
			$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print_hide_content.css", "Modules/Test"), "print");
		}

		if( $this->isPdfDeliveryRequest() )
		{
			require_once 'class.ilTestPDFGenerator.php';
			ilTestPDFGenerator::generatePDF($tpl->get(), ilTestPDFGenerator::PDF_OUTPUT_DOWNLOAD, $this->object->getTitle());
		}
		else
		{
			$this->tpl->setContent($tpl->get());
		}
	}

	/**
	 * Output of the pass overview for a test called by a test participant
	 *
	 * @global ilTabsGUI $ilTabs
	 */
	function outUserResultsOverview()
	{
		global $ilUser, $ilTabs;

		$ilTabs->setBackTarget(
			$this->lng->txt('tst_results_back_introduction'),
			$this->ctrl->getLinkTargetByClass('ilObjTestGUI', 'infoScreen')
		);

		$testSession = $this->testSessionFactory->getSession();
		$active_id = $testSession->getActiveId();
		$user_id = $ilUser->getId();
		$uname = $this->object->userLookupFullName($user_id, TRUE);

		if( !$this->object->canShowTestResults($testSession, $ilUser->getId()) )
		{
			$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
		}

		$templatehead = new ilTemplate("tpl.il_as_tst_results_participants.html", TRUE, TRUE, "Modules/Test");
		$template = new ilTemplate("tpl.il_as_tst_results_participant.html", TRUE, TRUE, "Modules/Test");

		require_once 'Modules/Test/classes/toolbars/class.ilTestResultsToolbarGUI.php';
		$toolbar = new ilTestResultsToolbarGUI($this->ctrl, $this->tpl, $this->lng);

		$this->ctrl->setParameter($this, 'pdf', '1');
		$toolbar->setPdfExportLinkTarget( $this->ctrl->getLinkTarget($this, 'outUserResultsOverview') );
		$this->ctrl->setParameter($this, 'pdf', '');

		include_once './Services/WebServices/RPC/classes/class.ilRPCServerSettings.php';
		if( $this->object->canShowCertificate($testSession, $user_id, $active_id) )
		{
			$toolbar->setCertificateLinkTarget($this->ctrl->getLinkTarget($this, 'outCertificate'));
		}

		$toolbar->build();
		
		$templatehead->setVariable('RESULTS_TOOLBAR', $this->ctrl->getHTML($toolbar));

		$hide_details = !$this->object->getShowPassDetails();
		#if ($hide_details)
		#{
		#	$executable = $this->object->isExecutable($testSession, $ilUser->getId());
		#	if (!$executable["executable"]) $hide_details = FALSE;
		#}

		$template->setCurrentBlock("pass_overview");
		$overview = $this->getPassOverview($active_id, "iltestevaluationgui", "outUserPassDetails", FALSE, $hide_details);
		$template->setVariable("PASS_OVERVIEW", $overview);
		$template->setVariable("TEXT_RESULTS", $this->lng->txt("tst_results_overview"));
		$template->parseCurrentBlock();

		$user_data = $this->getResultsUserdata($testSession, $active_id, TRUE);

		if ($this->object->getAnonymity()) {
			$template->setVariable("TEXT_HEADING", $this->lng->txt("tst_result"));
		}
		else {
			$template->setVariable("TEXT_HEADING", sprintf($this->lng->txt("tst_result_user_name"), $uname));
			$template->setVariable("USER_DATA", $user_data);
		}
		
		if( $this->isGradingMessageRequired() )
		{
			$template->setCurrentBlock('grading_message');
			$template->setVariable('GRADING_MESSAGE', $this->getGradingMessage($active_id));
			$template->parseCurrentBlock();
		}

		$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print.css", "Modules/Test"), "print");
		if ($this->object->getShowSolutionAnswersOnly())
		{
			$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print_hide_content.css", "Modules/Test"), "print");
		}
		$templatehead->setVariable("RESULTS_PARTICIPANT", $template->get());

		if( $this->isPdfDeliveryRequest() )
		{
			//$this->object->deliverPDFfromHTML($template->get(), $this->object->getTitle());
			require_once 'class.ilTestPDFGenerator.php';
			ilTestPDFGenerator::generatePDF($template->get(), ilTestPDFGenerator::PDF_OUTPUT_DOWNLOAD, $this->object->getTitle());
			//$this->object->deliverPDFfromHTML($template->get(), sprintf($this->lng->txt("tst_result_user_name"), $uname));
		}
		else
		{
			$this->tpl->setContent($templatehead->get());
		}
	}

	/**
	* Output of the pass overview for a user when he/she wants to see his/her list of answers
	*
	* Output of the pass overview for a user when he/she wants to see his/her list of answers
	*
	* @access public
	*/
	function outUserListOfAnswerPasses()
	{
		global $ilUser;

		if (!$this->object->getShowSolutionPrintview())
		{
			ilUtil::sendInfo($this->lng->txt("no_permission"), true);
			$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
		}

		$template = new ilTemplate("tpl.il_as_tst_info_list_of_answers.html", TRUE, TRUE, "Modules/Test");

		$pass = null;
		if (array_key_exists("pass", $_GET))
		{
			if (strlen($_GET["pass"])) $pass = $_GET["pass"];
		}
		$user_id = $ilUser->getId();
		
		$testSession = $this->testSessionFactory->getSession();
		$active_id = $testSession->getActiveId();
		
		$overview = "";
		if ($this->object->getNrOfTries() == 1)
		{
			$pass = 0;
		}
		else
		{
			$overview = $this->getPassOverview($active_id, "iltestevaluationgui", "outUserListOfAnswerPasses", TRUE);
			$template->setVariable("TEXT_RESULTS", $this->lng->txt("tst_passes"));
			$template->setVariable("PASS_OVERVIEW", $overview);
		}

		$signature = "";
		if (strlen($pass))
		{
			$signature = $this->getResultsSignature();
			$result_array =& $this->object->getTestResult($active_id, $pass);
			$user_id =& $this->object->_getUserIdFromActiveId($active_id);
			$showAllAnswers = TRUE;
			if ($this->object->isExecutable($testSession, $user_id))
			{
				$showAllAnswers = FALSE;
			}
			$answers = $this->getPassListOfAnswers($result_array, $active_id, $pass, FALSE, $showAllAnswers);
			$template->setVariable("PASS_DETAILS", $answers);
		}
		$template->setVariable("FORMACTION", $this->ctrl->getFormAction($this));
		$template->setVariable("BACK_TEXT", $this->lng->txt("tst_results_back_introduction"));
		$template->setVariable("BACK_URL", $this->ctrl->getLinkTargetByClass("ilobjtestgui", "infoScreen"));
		$template->setVariable("PRINT_TEXT", $this->lng->txt("print"));
		$template->setVariable("PRINT_URL", "javascript:window.print();");

		$user_data = $this->getResultsUserdata($testSession, $active_id, TRUE);
		$template->setVariable("USER_DATA", $user_data);
		$template->setVariable("TEXT_LIST_OF_ANSWERS", $this->lng->txt("tst_list_of_answers"));
		if (strlen($signature))
		{
			$template->setVariable("SIGNATURE", $signature);
		}
		if (!is_null($pass) && $this->object->isShowExamIdInTestResultsEnabled())
		{
			$template->setCurrentBlock('exam_id_footer');
			$template->setVariable('EXAM_ID_VAL', $this->object->lookupExamId(
				$testSession->getActiveId(), $pass
			));
			$template->setVariable('EXAM_ID_TXT', $this->lng->txt('exam_id'));
			$template->parseCurrentBlock();
		}
		$this->tpl->setVariable("ADM_CONTENT", $template->get());

		$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print.css", "Modules/Test"), "print");
		if ($this->object->getShowSolutionAnswersOnly())
		{
			$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print_hide_content.css", "Modules/Test"), "print");
		}
	}

	/**
	* Output of the learners view of an existing test pass
	*
	* Output of the learners view of an existing test pass
	*
	* @access public
	*/
	function passDetails()
	{
		if (array_key_exists("pass", $_GET) && (strlen($_GET["pass"]) > 0))
		{
			$this->ctrl->saveParameter($this, "pass");
			$this->ctrl->saveParameter($this, "active_id");
			$this->outTestResults(false, $_GET["pass"]);
		}
		else
		{
			$this->outTestResults(false);
		}
	}

	/**
	* Creates an output of the solution of an answer compared to the correct solution
	*
	* @access public
	*/
	function outCorrectSolution()
	{
		if( !$this->object->getShowSolutionDetails() )
		{
			ilUtil::sendInfo($this->lng->txt("no_permission"), true);
			$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
		}

		$testSession = $this->testSessionFactory->getSession();
		$activeId = $testSession->getActiveId();
		
		if( !($activeId > 0) )
		{
			$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
		}

		if( !$this->object->canShowTestResults($testSession, $testSession->getUserId()) )
		{
			$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
		}

		$this->ctrl->saveParameter($this, "pass");
		$pass = (int)$_GET['pass'];

		$testSequence = $this->testSequenceFactory->getSequenceByPass($testSession, $pass);
		$testSequence->loadFromDb();
		$testSequence->loadQuestions();

		$questionId = (int)$_GET['evaluation'];
		
		if( !$testSequence->questionExists($questionId) )
		{
			$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
		}
		
		global $ilTabs;

		$ilTabs->setBackTarget($this->lng->txt("tst_back_to_pass_details"), $this->ctrl->getLinkTarget($this, 'outUserPassDetails'));

		include_once("./Services/Style/classes/class.ilObjStyleSheet.php");
		$this->tpl->setCurrentBlock("ContentStyle");
		$this->tpl->setVariable("LOCATION_CONTENT_STYLESHEET", ilObjStyleSheet::getContentStylePath(0));
		$this->tpl->parseCurrentBlock();

		$this->tpl->setCurrentBlock("SyntaxStyle");
		$this->tpl->setVariable("LOCATION_SYNTAX_STYLESHEET", ilObjStyleSheet::getSyntaxStylePath());
		$this->tpl->parseCurrentBlock();

		$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print.css", "Modules/Test"), "print");
		if ($this->object->getShowSolutionAnswersOnly())
		{
			$this->tpl->addCss(ilUtil::getStyleSheetLocation("output", "test_print_hide_content.css", "Modules/Test"), "print");
		}

		$solution = $this->getCorrectSolutionOutput($questionId, $activeId, $pass);

		$this->tpl->setContent($solution);
	}

	/**
	 * Creates user results for single questions
	 *
	 */
	public function singleResults()
	{
		global $ilAccess;

		if ((!$ilAccess->checkAccess("tst_statistics", "", $this->ref_id)) && (!$ilAccess->checkAccess("write", "", $this->ref_id)))
		{
			// allow only evaluation access
			ilUtil::sendInfo($this->lng->txt("cannot_edit_test"), true);
			$this->ctrl->redirectByClass("ilobjtestgui", "infoScreen");
		}

		$data =& $this->object->getCompleteEvaluationData();
		$color_class = array("tblrow1", "tblrow2");
		$counter = 0;
		$this->tpl->addBlockFile("ADM_CONTENT", "adm_content", "tpl.il_as_tst_eval_single_answers.html", "Modules/Test");
		$foundParticipants =& $data->getParticipants();
		if (count($foundParticipants) == 0)
		{
			ilUtil::sendInfo($this->lng->txt("tst_no_evaluation_data"));
			return;
		}
		else
		{
			$rows = array();
			foreach ($data->getQuestionTitles() as $question_id => $question_title)
			{
				$answered = 0;
				$reached = 0;
				$max = 0;
				foreach ($foundParticipants as $userdata)
				{
					$pass = $userdata->getScoredPass();
					if (is_object($userdata->getPass($pass)))
					{
						$question =& $userdata->getPass($pass)->getAnsweredQuestionByQuestionId($question_id);
						if (is_array($question))
						{
							$answered++;
						}
					}
				}
				$counter++;
				$this->ctrl->setParameter($this, "qid", $question_id);
				require_once './Modules/TestQuestionPool/classes/class.assQuestion.php';
				$question_object = assQuestion::_instanciateQuestion($question_id);
				$download = "";
				if ( $question_object instanceof ilObjFileHandlingQuestionType )
				{
					if ($question_object->hasFileUploads($this->object->getTestId()))
					{
						$download = "<a href=\"" . $this->ctrl->getLinkTarget($this, "exportFileUploadsForAllParticipants"). "\">" . $this->lng->txt("download") . "</a>";
					}
				}
				array_push($rows, 
					array(
						'qid'               => $question_id,
						'question_title'    => $question_title,
						'number_of_answers' => $answered,
						'output'            => "<a href=\"" . $this->ctrl->getLinkTarget($this, "exportQuestionForAllParticipants") . "\">" . $this->lng->txt("pdf_export") . "</a>",
						'file_uploads'      => $download
					)
				);
			}
			if (count($rows))
			{
				require_once './Modules/Test/classes/tables/class.ilResultsByQuestionTableGUI.php';
				$table_gui = new ilResultsByQuestionTableGUI($this, "singleResults");
				$table_gui->setTitle($this->lng->txt("tst_answered_questions_test"));
				$table_gui->setData($rows);

				$this->tpl->setVariable("TBL_SINGLE_ANSWERS", $table_gui->getHTML());
			}
			else
			{
				$this->tpl->setVariable("TBL_SINGLE_ANSWERS", $this->lng->txt("adm_no_special_users"));
			}
		}
	}

	/**
	* Output of a test certificate
	*/
	public function outCertificate()
	{
		$testSession = $this->testSessionFactory->getSession();
		
		require_once './Services/Certificate/classes/class.ilCertificate.php';
		require_once './Modules/Test/classes/class.ilTestCertificateAdapter.php';
		$certificate = new ilCertificate(new ilTestCertificateAdapter( $this->object ) );
		$certificate->outCertificate(
			array(
				"active_id" => $testSession->getActiveId(), 
				"pass" 		=> ilObjTest::_getResultPass( $testSession->getActiveId() ) 
			)
		);
	}
	
	public function confirmDeletePass()
	{
		if( !$this->object->isPassDeletionAllowed() )
		{
			$this->ctrl->redirect($this, 'outUserResultsOverview');
		}

		require_once 'Modules/Test/classes/confirmations/class.ilTestPassDeletionConfirmationGUI.php';

		if( isset($_GET['context']) && strlen($_GET['context']) )
		{
			$context = $_GET['context'];
		}
		else
		{
			$context = ilTestPassDeletionConfirmationGUI::CONTEXT_PASS_OVERVIEW;
		}

		$confirm = new ilTestPassDeletionConfirmationGUI($this->ctrl, $this->lng, $this);
		$confirm->build((int)$_GET['active_id'], (int)$_GET['pass'], $context);

		global $tpl;
		$tpl->setContent($this->ctrl->getHTML($confirm));
	}

	public function cancelDeletePass()
	{
		$this->redirectToPassDeletionContext($_POST['context']);
	}
	
	private function redirectToPassDeletionContext($context)
	{
		require_once 'Modules/Test/classes/confirmations/class.ilTestPassDeletionConfirmationGUI.php';

		switch($context)
		{
			case ilTestPassDeletionConfirmationGUI::CONTEXT_PASS_OVERVIEW:

				$this->ctrl->redirect($this, 'outUserResultsOverview');

			case ilTestPassDeletionConfirmationGUI::CONTEXT_INFO_SCREEN:

				$this->ctrl->redirectByClass('ilObjTestGUI', 'infoScreen');

			case ilTestPassDeletionConfirmationGUI::CONTEXT_DYN_TEST_PLAYER:

				$this->ctrl->redirectByClass('ilTestPlayerDynamicQuestionSetGUI', 'startTest');
		}
	}
	
	public function performDeletePass()
	{
		if( !$this->object->isPassDeletionAllowed() )
		{
			$this->ctrl->redirect($this, 'outUserResultsOverview');
		}
			/** @var ilDB $ilDB */
			global $ilDB;

		$active_fi = null;
		$pass = null;

		if( isset($_POST['active_id']) && (int)$_POST['active_id'] )
		{
			$active_fi = $_POST['active_id'];
		}

		if( isset($_POST['pass']) && is_numeric($_POST['pass']) )
		{
			$pass = $_POST['pass'];
		}

		if( is_null($active_fi) || is_null($pass) )
		{
			$this->ctrl->redirect($this, 'outUserResultsOverview');
		}

		if( !$this->object->isDynamicTest() && $pass == $this->object->_getResultPass($active_fi) )
		{
			$this->ctrl->redirect($this, 'outUserResultsOverview');
		}
			
			// Get information
			$result = $ilDB->query("
				SELECT tst_active.tries, tst_active.last_finished_pass, tst_sequence.pass
				FROM tst_active
				LEFT JOIN tst_sequence
				ON tst_sequence.active_fi = tst_active.active_id
				AND tst_sequence.pass = tst_active.tries
				WHERE tst_active.active_id = {$ilDB->quote($active_fi, 'integer')}
			");
		
			$row = $ilDB->fetchAssoc($result);
			
			$tries = $row['tries'];
			$lastFinishedPass = is_numeric($row['last_finished_pass']) ? $row['last_finished_pass'] : -1;
		
			if( $pass < $lastFinishedPass )
			{
				$isActivePass = false;
				$must_renumber = true;
			}
			elseif( $pass == $lastFinishedPass )
			{
				$isActivePass = false;
				
				if( $tries == $row['pass'] )
				{
					$must_renumber = true;
				}
				else
				{
					$must_renumber = false;
				}
			}
			elseif( $pass == $row['pass'] )
			{
				$isActivePass = true;
				$must_renumber = false;
			}
			else
			{
				throw new ilTestException ('This should not happen, please contact Bjoern Heyser to clean up this pass salad!');
			}

		if( !$this->object->isDynamicTest() && $isActivePass )
		{
			$this->ctrl->redirect($this, 'outUserResultsOverview');
		}
		
			if( $pass == 0 && (
					($lastFinishedPass == 0 && $tries == 1 && $tries != $row['pass'])
					|| ($isActivePass == true) // should be equal to || ($lastFinishedPass == -1 && $tries == 0)
				))
			{
				$last_pass = true;
			}
			else
			{
				$last_pass = false;
			}
						
			// Work on tables:
			// tst_active
			if ($last_pass)
			{
				$ilDB->manipulate(
					'DELETE
					FROM tst_active
					WHERE active_id = ' . $ilDB->quote($active_fi, 'integer')
				);
			}
			elseif( !$isActivePass )
			{
				$ilDB->manipulate(
					'UPDATE tst_active
					SET tries = ' . $ilDB->quote($tries-1, 'integer') . ',
					last_finished_pass = ' . $ilDB->quote($lastFinishedPass-1, 'integer') . '
					WHERE active_id = ' . $ilDB->quote($active_fi, 'integer')
				);
			}
			// tst_manual_fb
			$ilDB->manipulate(
				'DELETE
				FROM tst_manual_fb
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer') . '
				AND pass = ' . $ilDB->quote($pass, 'integer')
			);
			
			if ($must_renumber)
			{
				$ilDB->manipulate(
				'UPDATE tst_manual_fb
				SET pass = pass - 1
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer'). '
				AND pass > ' . $ilDB->quote($pass, 'integer')
				); 
			}
			
			// tst_mark -> nothing to do
			// 
			// tst_pass_result
			$ilDB->manipulate(
				'DELETE
				FROM tst_pass_result
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer') . '
				AND pass = ' . $ilDB->quote($pass, 'integer')
			);
			
			if ($must_renumber)
			{
				$ilDB->manipulate(
				'UPDATE tst_pass_result
				SET pass = pass - 1
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer'). '
				AND pass > ' . $ilDB->quote($pass, 'integer')
				); 
			}			
			
			// tst_qst_solved -> nothing to do
			
			// tst_rnd_copy -> nothing to do
			// tst_rnd_qpl_title -> nothing to do
			
			// tst_sequence
			$ilDB->manipulate(
				'DELETE
				FROM tst_sequence
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer') . '
				AND pass = ' . $ilDB->quote($pass, 'integer')
			);
			
			if ($must_renumber)
			{
				$ilDB->manipulate(
				'UPDATE tst_sequence
				SET pass = pass - 1
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer'). '
				AND pass > ' . $ilDB->quote($pass, 'integer')
				); 
			}
		
		if( $this->object->isDynamicTest() )
		{
			$tables = array(
				'tst_seq_qst_tracking', 'tst_seq_qst_answstatus', 'tst_seq_qst_postponed', 'tst_seq_qst_checked'
			);
			
			foreach($tables as $table)
			{
				$ilDB->manipulate("
						DELETE FROM $table
						WHERE active_fi = {$ilDB->quote($active_fi, 'integer')}
						AND pass = {$ilDB->quote($pass, 'integer')}
				");
				
				if( $must_renumber )
				{
					$ilDB->manipulate("
						UPDATE $table
						SET pass = pass - 1
						WHERE active_fi = {$ilDB->quote($active_fi, 'integer')}
						AND pass > {$ilDB->quote($pass, 'integer')}
					");
				}
			}
		}
						
			// tst_solutions
			$ilDB->manipulate(
				'DELETE
				FROM tst_solutions
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer') . '
				AND pass = ' . $ilDB->quote($pass, 'integer')
			);
			
			if ($must_renumber)
			{
				$ilDB->manipulate(
				'UPDATE tst_solutions
				SET pass = pass - 1
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer'). '
				AND pass > ' . $ilDB->quote($pass, 'integer')
				); 
			}		

			// tst_test_result
			$ilDB->manipulate(
				'DELETE
				FROM tst_test_result
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer') . '
				AND pass = ' . $ilDB->quote($pass, 'integer')
			);
			
			if ($must_renumber)
			{
				$ilDB->manipulate(
				'UPDATE tst_test_result
				SET pass = pass - 1
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer') . '
				AND pass > ' . $ilDB->quote($pass, 'integer')
				);
			}		
			
			// tst_test_rnd_qst -> nothing to do
			
			// tst_times
			$ilDB->manipulate(
				'DELETE
				FROM tst_times
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer') . '
				AND pass = ' . $ilDB->quote($pass, 'integer')
			);
			
			if ($must_renumber)
			{
				$ilDB->manipulate(
				'UPDATE tst_times
				SET pass = pass - 1
				WHERE active_fi = ' . $ilDB->quote($active_fi, 'integer'). '
				AND pass > ' . $ilDB->quote($pass, 'integer')
				); 
			}
			
			require_once 'Modules/Test/classes/class.ilObjAssessmentFolder.php';
			if (ilObjAssessmentFolder::_enabledAssessmentLogging())
			{
				$this->object->logAction($this->lng->txtlng("assessment", "log_deleted_pass", ilObjAssessmentFolder::_getLogLanguage()));
			}
			// tst_result_cache
			// Ggfls. nur renumbern.
			require_once './Modules/TestQuestionPool/classes/class.assQuestion.php';
			assQuestion::_updateTestResultCache($active_fi);
		
		if( $this->object->isDynamicTest() )
		{
			require_once 'Modules/Test/classes/tables/class.ilTestDynamicQuestionSetStatisticTableGUI.php';
			unset($_SESSION['form_'.ilTestDynamicQuestionSetStatisticTableGUI::FILTERED_TABLE_ID]);
		}

		$this->redirectToPassDeletionContext($_POST['context']);
	}

	protected function getFilteredTestResult($active_id, $pass)
	{
		global $ilDB, $ilPluginAdmin;

		$testResults = $this->object->getTestResult($active_id, $pass);
		$questionIds = array();
		foreach($testResults as $resultItemKey => $resultItemValue)
		{
			if($resultItemKey === 'test' || $resultItemKey === 'pass')
			{
				continue;
			}

			$questionIds[] = $resultItemValue['qid'];
		}

		$table_gui = $this->buildPassDetailsOverviewTableGUI($this, 'outUserPassDetails');
		$table_gui->initFilter();

		require_once 'Modules/TestQuestionPool/classes/class.ilAssQuestionList.php';
		$questionList = new ilAssQuestionList($ilDB, $this->lng, $ilPluginAdmin, null);
		$questionList->setQuestionIdsFilter($questionIds);
		$questionList->setQuestionInstanceTypeFilter(null);

		foreach ($table_gui->getFilterItems() as $item)
		{
			if( substr($item->getPostVar(), 0, strlen('tax_')) == 'tax_' )
			{
				$v = $item->getValue();

				if( is_array($v) && count($v) && !(int)$v[0] )
				{
					continue;
				}

				$taxId = substr($item->getPostVar(), strlen('tax_'));
				$questionList->addTaxonomyFilter($taxId, $item->getValue());
			}
			elseif( $item->getValue() !== false )
			{
				$questionList->addFieldFilter($item->getPostVar(), $item->getValue());
			}
		}

		$questionList->load();

		$filteredTestResult = array();

		foreach($testResults as $resultItemKey => $resultItemValue)
		{
			if($resultItemKey === 'test' || $resultItemKey === 'pass')
			{
				continue;
			}

			if( !$questionList->isInList($resultItemValue['qid']) )
			{
				continue;
			}

			$filteredTestResult[] = $resultItemValue;
		}

		return $filteredTestResult;
	}
}
